#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Phase 4: KFX to EPUB Conversion
Converts imported books to EPUB format
"""

import os
import sys
import subprocess
import threading
import time


def query_book_info_from_db(library_path, book_ids):
    """Query Calibre database for book info"""
    import sqlite3
    from modules.utils import print_error
    
    db_path = os.path.join(library_path, "metadata.db")
    book_info = {}
    
    try:
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        
        for book_id in book_ids:
            cursor.execute("SELECT title, path FROM books WHERE id = ?", (book_id,))
            result = cursor.fetchone()
            
            if result:
                title = result[0]
                path = result[1]
                
                cursor.execute("""
                    SELECT authors.name 
                    FROM authors 
                    JOIN books_authors_link ON authors.id = books_authors_link.author
                    WHERE books_authors_link.book = ?
                    LIMIT 1
                """, (book_id,))
                author_result = cursor.fetchone()
                author = author_result[0] if author_result else "Unknown Author"
                
                book_info[book_id] = {
                    'title': title,
                    'author': author,
                    'path': path
                }
        
        conn.close()
        return book_info
        
    except Exception as e:
        print_error(f"Failed to query database: {e}")
        return {}


def find_source_file_in_directory(book_dir):
    """Find the source ebook file in the book directory"""
    from modules.utils import print_error
    
    try:
        if not os.path.exists(book_dir):
            return None, False
        
        for filename in os.listdir(book_dir):
            lower_name = filename.lower()
            if lower_name.endswith(('.kfx', '.azw', '.azw3', '.kfx-zip', '.mobi')):
                is_kfx_zip = lower_name.endswith('.kfx-zip')
                return filename, is_kfx_zip
        
        return None, False
        
    except Exception as e:
        print_error(f"Error searching directory {book_dir}: {e}")
        return None, False


def convert_book_to_epub(source_path, epub_path, timeout_seconds=180, raw_log_path=None, book_info=None):
    """Convert book to EPUB using ebook-convert with RAW logging"""
    from modules.utils import display_progress_timer, append_to_raw_log
    
    cmd = [
        'ebook-convert',
        source_path,
        epub_path,
        '--input-profile=default',
        '--output-profile=tablet',
        '--no-svg-cover',
        '--epub-version=3'
    ]
    
    try:
        process = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding='utf-8',
            errors='replace'
        )
        
        timer_stopped = threading.Event()
        
        timer_thread = threading.Thread(
            target=display_progress_timer,
            args=("Converting to EPUB", timeout_seconds, timer_stopped),
            daemon=True
        )
        timer_thread.start()
        
        try:
            stdout, stderr = process.communicate(timeout=timeout_seconds)
            result_returncode = process.returncode
            timer_stopped.set()
            timer_thread.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
        except subprocess.TimeoutExpired:
            timer_stopped.set()
            timer_thread.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
            process.kill()
            stdout, stderr = process.communicate()
            result_returncode = -1
        
        # Log to RAW log if enabled
        if raw_log_path:
            append_to_raw_log(
                raw_log_path,
                cmd,
                stdout,
                stderr,
                result_returncode,
                book_info=book_info
            )
        
        if result_returncode == 0 and os.path.exists(epub_path):
            return True, "", {'is_timeout': False}
        else:
            error_msg = stderr if stderr else "Conversion failed"
            return False, error_msg, {'is_timeout': False}
            
    except subprocess.TimeoutExpired:
        # Log timeout to RAW log if enabled
        if raw_log_path:
            append_to_raw_log(
                raw_log_path,
                cmd,
                '',
                '',
                -1,  # Timeout exit code
                book_info=book_info
            )
        return False, "Conversion timeout", {'is_timeout': True}
    except Exception as e:
        return False, str(e), {'is_timeout': False}


def add_epub_format_to_calibre(book_id, epub_path, library_path):
    """Add EPUB format to existing Calibre book record"""

    cmd = [
        'calibredb', 'add_format',
        str(book_id),
        epub_path,
        f'--library-path={library_path}'
    ]

    try:
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=60)

        if result.returncode == 0:
            return True, ""
        else:
            error_msg = result.stderr if result.stderr else "Failed to add format"
            return False, error_msg
            
    except subprocess.TimeoutExpired:
        return False, "Add format timeout"
    except Exception as e:
        return False, str(e)


def remove_format_from_calibre(book_id, format_name, library_path):
    """
    Remove a specific format from Calibre book record using calibredb
    Returns: (success: bool, error_message: str)
    """
    try:
        cmd = [
            'calibredb', 'remove_format',
            str(book_id),
            format_name.upper(),
            f'--library-path={library_path}'
        ]
        
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=60)
        
        if result.returncode == 0:
            return True, ""
        else:
            error_msg = result.stderr if result.stderr else "Failed to remove format"
            return False, error_msg
            
    except subprocess.TimeoutExpired:
        return False, "Remove format timeout"
    except Exception as e:
        return False, str(e)


def run_phase_4(calibre_path, book_ids, calibre_config=None, working_dir=None):
    """
    Run Phase 4: KFX to EPUB Conversion
    
    Args:
        calibre_path: Calibre library path
        book_ids: List of book IDs from Phase 3
        calibre_config: Calibre configuration dict (optional)
        working_dir: Working directory for logs (optional)
    
    Returns:
        dict with keys:
            - success: bool
            - converted: int
            - merged: int
            - failed: int
            - stats: dict with conversion statistics
    """
    from modules.utils import print_step, print_ok, print_error, print_warn, append_to_failed_books
    
    # Get config settings
    source_file_management = 'keep_both'
    if calibre_config and isinstance(calibre_config, dict):
        source_file_management = calibre_config.get('source_file_management', 'keep_both')
    
    stats = {
        'total': len(book_ids),
        'converted': 0,
        'merged': 0,
        'failed': 0,
        'errors': [],
        'failed_conversions': [],
        'source_files_deleted': 0
    }
    
    if not book_ids:
        print_warn("No books to convert")
        return {
            'success': False,
            'converted': 0,
            'merged': 0,
            'failed': 0,
            'stats': stats
        }
    
    print_step(f"Converting {len(book_ids)} book(s) to EPUB format...")
    print()
    print_warn("NOTE: Each book has a 3-minute timeout for conversion")
    print()
    
    # Query database for book info
    print_step("Querying Calibre database for book information...")
    book_info = query_book_info_from_db(calibre_path, book_ids)
    
    if not book_info:
        print_error("Failed to retrieve book information from database")
        return {
            'success': False,
            'converted': 0,
            'merged': 0,
            'failed': 0,
            'stats': stats
        }
    
    print_ok(f"Retrieved information for {len(book_info)} book(s)")
    print()
    
    # Initialize RAW log path if enabled
    from modules.utils import get_raw_log_path, cleanup_old_raw_logs
    from modules.config import load_config
    
    config = load_config()
    enable_raw_logs = config.get('enable_raw_logs', False) if config else False
    raw_log_path = None
    if enable_raw_logs and working_dir:
        raw_log_path = get_raw_log_path(working_dir, 'conversion')
        conversion_logs_dir = os.path.join(working_dir, "Logs", "03_Conversion")
        cleanup_old_raw_logs(conversion_logs_dir, keep_count=10)
    
    # Process each book
    for idx, book_id in enumerate(book_ids, 1):
        if book_id not in book_info:
            continue
            
        info = book_info[book_id]
        title = info['title']
        author = info['author']
        book_path = info['path']
        
        print_step(f"Processing book {idx}/{len(book_info)}: '{title}' by {author}")
        
        book_dir = os.path.join(calibre_path, book_path)
        
        source_filename, is_kfx_zip = find_source_file_in_directory(book_dir)
        
        if not source_filename:
            error_msg = f"No source file found in {book_path}"
            print_error(error_msg)
            stats['failed'] += 1
            stats['errors'].append(f"Book {book_id}: {error_msg}")
            stats['failed_conversions'].append((book_id, title, author, error_msg))
            
            # Track failed conversion in Failed-Books.txt
            if working_dir:
                append_to_failed_books(working_dir, title, str(book_id))
            
            print()
            continue
        
        source_path = os.path.join(book_dir, source_filename)
        epub_filename = os.path.splitext(source_filename)[0] + '.epub'
        epub_path = os.path.join(book_dir, epub_filename)
        
        print(f"  Source: {source_filename}")
        print(f"  Target: {epub_filename}")
        print("  Converting to EPUB...", end='', flush=True)
        
        # Create book info string for logging
        book_info_str = f"{title} by {author} (ID: {book_id})"
        
        success, error, error_type = convert_book_to_epub(
            source_path, 
            epub_path, 
            timeout_seconds=180,
            raw_log_path=raw_log_path,
            book_info=book_info_str
        )
        
        if not success:
            if error_type.get('is_timeout'):
                print_error(f"  TIMEOUT")
            else:
                print_error(f"  FAILED")
            print_error(f"  {error}")
            stats['failed'] += 1
            stats['errors'].append(f"Book {book_id}: {error}")
            stats['failed_conversions'].append((book_id, title, author, error))
            
            # Track failed conversion in Failed-Books.txt
            if working_dir:
                append_to_failed_books(working_dir, title, str(book_id))
            
            print()
            continue
        
        print_ok("  Conversion successful")
        stats['converted'] += 1
        
        # Merge EPUB format into Calibre
        print("  Merging EPUB format to Calibre...", end='', flush=True)
        
        success, error = add_epub_format_to_calibre(book_id, epub_path, calibre_path)
        
        if not success:
            print_error(f"  {error}")
        else:
            print_ok("  EPUB format merged successfully")
            stats['merged'] += 1
        
        # Handle source file management based on user choice
        # Use calibredb remove_format to properly remove from Calibre database
        if source_file_management == 'delete_originals':
            # Delete the original source format after successful conversion
            # Determine the actual source format extension (e.g., KFX, AZW3, MOBI)
            source_ext = os.path.splitext(source_filename)[1].upper().replace('.', '')
            
            # Only delete KFX and KFX-ZIP formats (preserve MOBI/AZW/AZW3)
            if source_ext in ['KFX', 'KFX-ZIP']:
                print(f"  Removing {source_ext} format from Calibre...", end='', flush=True)
                success, error = remove_format_from_calibre(book_id, source_ext, calibre_path)
                if success:
                    print_ok(f"  {source_ext} format removed from Calibre")
                    stats['source_files_deleted'] += 1
                else:
                    print_warn(f"  Failed to remove {source_ext} format: {error}")
            else:
                print(f"  Keeping {source_ext} format (only KFX/KFX-ZIP are deleted)")
        
        elif source_file_management == 'delete_kfx_zip_only' and is_kfx_zip:
            # Delete only .kfx-zip files
            print("  Removing KFX-ZIP format from Calibre...", end='', flush=True)
            success, error = remove_format_from_calibre(book_id, 'KFX-ZIP', calibre_path)
            if success:
                print_ok("  KFX-ZIP format removed from Calibre")
                stats['source_files_deleted'] += 1
            else:
                print_warn(f"  Failed to remove KFX-ZIP format: {error}")
        
        print()
    
    # Summary
    print("--------------------------------------------------")
    print_step("Conversion Summary:")
    print()
    print_ok(f"Total books processed: {stats['total']}")
    print_ok(f"Successfully converted: {stats['converted']}")
    print_ok(f"Successfully merged to Calibre: {stats['merged']}")
    
    if stats['failed'] > 0:
        print_error(f"Failed conversions: {stats['failed']}")
    
    print()
    
    return {
        'success': stats['converted'] > 0,
        'converted': stats['converted'],
        'merged': stats['merged'],
        'failed': stats['failed'],
        'stats': stats
    }
