#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Utilities module - Contains logging, printing, and helper functions
"""

import os
import sys
import json
import subprocess
import shutil
import time
import threading
import platform
from datetime import datetime

# Add parent to path for imports (check both root and code/ subfolder)
script_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, script_dir)

# Check if code folder exists (for reorganized structure)
code_dir = os.path.join(script_dir, "code")
if os.path.exists(code_dir):
    sys.path.insert(0, code_dir)

# Script Version - Defined here to avoid circular import issues
SCRIPT_VERSION = "2026.02.28.JH"
KFXKEYEXTRACTOR_VERSION = "v10.0.17"

# Unified Configuration File
CONFIG_FILE = "key_finder_config.json"

# Tool filename prefixes - used for case-insensitive, version-agnostic tool discovery.
# The script searches for any .exe whose lowercase name STARTS WITH this prefix.
# Drop a new version (e.g. KFXKeyExtractor30.exe) into code/tools/ and it will be
# found automatically on the next run - no code changes required.
KFXKEYEXTRACTOR_PREFIX = "kfxkeyextractor"
KFXARCHIVER_PREFIX = "kfxarchiver"


def print_colored(message: str, color: str) -> None:
    """Print colored messages"""
    colors = {
        'cyan': '\033[96m',
        'green': '\033[92m',
        'yellow': '\033[93m',
        'red': '\033[91m',
        'magenta': '\033[95m',
        'end': '\033[0m'
    }
    print(f"{colors.get(color, '')}{message}{colors['end']}")


def print_step(message: str) -> None:
    print_colored(f"[*] {message}", 'cyan')


def print_ok(message: str) -> None:
    print_colored(f"[OK] {message}", 'green')


def print_warn(message: str) -> None:
    print_colored(f"[!] {message}", 'yellow')


def print_error(message: str) -> None:
    print_colored(f"[ERROR] {message}", 'red')


def print_done(message: str) -> None:
    print_colored(f"[DONE] {message}", 'magenta')


def parse_version(version_str):
    """
    Parse version string into comparable components.
    Format: YYYY.MM.DD[.build]JH
    
    Args:
        version_str: Version string (e.g., "2025.11.11.JH" or "2025.11.11.1.JH")
    
    Returns:
        tuple: (year, month, day, build) or None if parsing fails
        - build defaults to 0 if not present
    """
    try:
        version_clean = version_str.replace('.JH', '').replace('JH', '')
        parts = version_clean.split('.')
        
        if len(parts) < 3:
            return None
        
        year = int(parts[0])
        month = int(parts[1])
        day = int(parts[2])
        build = int(parts[3]) if len(parts) >= 4 else 0
        
        if not (2000 <= year <= 2100 and 1 <= month <= 12 and 1 <= day <= 31):
            return None
        
        return (year, month, day, build)
        
    except (ValueError, IndexError):
        return None


def compare_versions(version1, version2):
    """
    Compare two version strings chronologically.
    """
    v1 = parse_version(version1)
    v2 = parse_version(version2)
    
    if v1 is None or v2 is None:
        return None
    
    if v1 < v2:
        return -1
    elif v1 > v2:
        return 1
    else:
        return 0


def check_latest_version():
    """
    Check for latest version from GitHub repository releases
    Returns: (latest_version: str or None, error_message: str or None)
    """
    try:
        import urllib.request
        import json
        
        url = "https://api.github.com/repos/jadehawk/Kindle_Key_Finder/releases/latest"
        req = urllib.request.Request(url, headers={'User-Agent': 'Mozilla/5.0'})
        
        with urllib.request.urlopen(req, timeout=5) as response:
            data = json.loads(response.read().decode('utf-8'))
            tag_name = data.get('tag_name', '')
            
            if tag_name:
                version = tag_name.lstrip('v')
                return version, None
            else:
                return None, "No tag_name found in latest release"
                
    except Exception as e:
        return None, str(e)


def print_banner_and_version():
    """Print ASCII banner and script version"""
    print_colored(r" _____         _                   _   _       _                  ____                ", 'green')
    print_colored(r"|_   _|__  ___| |__  _   _        | \ | | ___ | |_ ___  ___      / ___|___  _ __ ___  ", 'green')
    print_colored(r"  | |/ _ \/ __| '_ \| | | | _____ |  \| |/ _ \| __/ _ \/ __|    | |   / _ \| '_ ` _ \ ", 'green')
    print_colored(r"  | |  __/ (__| | | | |_| ||_____|| |\  | (_) | ||  __/\__ \  _ | |__| (_) | | | | | |", 'green')
    print_colored(r"  |_|\___|\___|_| |_|\__, |       |_| \_|\___/ \__\___||___/ (_) \____\___/|_| |_| |_|", 'green')
    print_colored(r"                     |___/                                                             ", 'green')
    print()
    print_step("Kindle Key Finder - Python Edition Wrapper")
    print(f"Script Version: {SCRIPT_VERSION}")
    print()
    
    print_step("Checking for updates...")
    latest_version, error = check_latest_version()
    
    if latest_version:
        comparison = compare_versions(SCRIPT_VERSION, latest_version)
        
        if comparison is None:
            print_colored("  (Unable to parse version - check manually)", 'cyan')
            print_colored("  Visit https://techy-notes.com for the latest version", 'cyan')
            print()
        elif comparison < 0:
            print()
            print_colored("=" * 70, 'yellow')
            print_colored(f"  ⚠   NEW VERSION AVAILABLE: {latest_version}", 'yellow')
            print_colored(f"  Current version: {SCRIPT_VERSION}", 'yellow')
            print_colored("  Visit https://techy-notes.com for the latest version", 'yellow')
            print_colored("=" * 70, 'yellow')
            print()
        elif comparison > 0:
            print_colored(f"  ℹ   You are running a development version ({SCRIPT_VERSION})", 'cyan')
            print_colored(f"  Latest stable release: {latest_version}", 'cyan')
            print()
        else:
            print_colored("  ✓   You are running the latest version", 'green')
            print()
    else:
        print_colored("  (Unable to check for updates - connection failed)", 'cyan')
        print_colored("  Visit https://techy-notes.com for the latest version", 'cyan')
        print()


def display_phase_banner(phase_num, phase_name):
    """Display a prominent phase banner with decorative borders"""
    print()
    print_colored("═" * 70, 'cyan')
    print_colored(f"║{f'PHASE {phase_num}':^68}║", 'cyan')
    print_colored(f"║{phase_name.upper():^68}║", 'cyan')
    print_colored(f"║{f'Script Version: {SCRIPT_VERSION}':^68}║", 'cyan')
    print_colored("═" * 70, 'cyan')
    print()


def obfuscate_sensitive(text):
    """
    Obfuscate sensitive strings by showing first 2 and last 2 characters
    """
    if len(text) <= 4:
        return text
    return text[:2] + '*' * (len(text) - 4) + text[-2:]


def filter_sensitive_output(text, hide_sensitive=False):
    """
    Filter and obfuscate sensitive information in output text
    """
    import re
    lines = text.split('\n')
    filtered_lines = []
    
    for line in lines:
        # Always skip Qt and Fontconfig error messages
        if 'QObject::startTimer: Timers can only be used with threads started with QThread' in line:
            continue
        if 'Fontconfig error: Cannot load default config file' in line:
            continue
        
        if not hide_sensitive:
            filtered_lines.append(line)
            continue
        
        filtered_line = line
        
        # Obfuscate DSN values
        if line.startswith('DSN '):
            dsn_value = line.replace('DSN ', '').strip()
            if dsn_value:
                filtered_line = f"DSN {obfuscate_sensitive(dsn_value)}"
        
        # Obfuscate Tokens
        elif line.startswith('Tokens '):
            tokens_value = line.replace('Tokens ', '').strip()
            if tokens_value:
                if ',' in tokens_value:
                    token_parts = tokens_value.split(',')
                    obfuscated_parts = [obfuscate_sensitive(part.strip()) for part in token_parts]
                    filtered_line = f"Tokens {','.join(obfuscated_parts)}"
                else:
                    filtered_line = f"Tokens {obfuscate_sensitive(tokens_value)}"
        
        # Obfuscate DRM key with UUID and secret key
        elif 'amzn1.drm-key.v1.' in line and '$secret_key:' in line:
            match = re.search(r'(amzn1\.drm-key\.v1\.)([a-f0-9\-]+)(\$secret_key:)([a-f0-9]+)', line)
            if match:
                prefix = match.group(1)
                uuid = match.group(2)
                middle = match.group(3)
                secret = match.group(4)
                
                obfuscated_uuid = obfuscate_sensitive(uuid)
                obfuscated_secret = obfuscate_sensitive(secret)
                
                original = f"{prefix}{uuid}{middle}{secret}"
                replacement = f"{prefix}{obfuscated_uuid}{middle}{obfuscated_secret}"
                filtered_line = line.replace(original, replacement)
        
        filtered_lines.append(filtered_line)
    
    return '\n'.join(filtered_lines)


def display_progress_timer(message, timeout_seconds, timer_stopped_event):
    """Reusable progress timer display function"""
    start_time = time.time()
    while not timer_stopped_event.is_set():
        elapsed = int(time.time() - start_time)
        minutes, seconds = divmod(elapsed, 60)
        timeout_mins, timeout_secs = divmod(timeout_seconds, 60)
        print(f"\r  {message}.. [ {minutes:02d}:{seconds:02d} / {timeout_mins:02d}:{timeout_secs:02d} ]", 
              end='', flush=True)
        time.sleep(1)


def get_disk_space(path):
    """Get disk space information for a given path"""
    try:
        check_path = path
        while not os.path.exists(check_path) and check_path != os.path.dirname(check_path):
            check_path = os.path.dirname(check_path)
        
        if not os.path.exists(check_path):
            return None, None, None
        
        stat = shutil.disk_usage(check_path)
        free_gb = stat.free / (1024 ** 3)
        total_gb = stat.total / (1024 ** 3)
        percent_free = (stat.free / stat.total) * 100 if stat.total > 0 else 0
        
        return free_gb, total_gb, percent_free
    except Exception as e:
        return None, None, None


def test_kindle_folder_accessibility(kindle_content_path, working_dir=None):
    """
    Test if Kindle content folder is accessible by performing actual file operations.
    This helps detect access violations (0xc0000005) before attempting key extraction.
    
    Args:
        kindle_content_path: Path to Kindle content directory
        working_dir: Working directory for temp files (optional)
    
    Returns:
        tuple: (success: bool, error_message: str or None, test_file: str or None)
    """
    import tempfile
    
    temp_test_dir = None
    
    try:
        # Find a book file to test with
        test_book_file = None
        
        if not os.path.exists(kindle_content_path):
            return False, f"Kindle content path does not exist: {kindle_content_path}", None
        
        # Search for any book file (.azw, .kfx, etc.)
        for root, dirs, files in os.walk(kindle_content_path):
            for file in files:
                if file.lower().endswith(('.azw', '.kfx', '.kfx-zip', '.azw3', '.azw1', '.mobi')):
                    test_book_file = os.path.join(root, file)
                    break
            if test_book_file:
                break
        
        if not test_book_file:
            return False, "No book files found in Kindle content directory", None
        
        # Create temp directory for test
        if working_dir:
            temp_test_dir = os.path.join(working_dir, "temp_access_test")
        else:
            temp_test_dir = os.path.join(tempfile.gettempdir(), "kindle_access_test")
        
        os.makedirs(temp_test_dir, exist_ok=True)
        
        # Get file info
        file_name = os.path.basename(test_book_file)
        file_size = os.path.getsize(test_book_file)
        temp_copy_path = os.path.join(temp_test_dir, file_name)
        
        # Test 1: Copy file
        try:
            shutil.copy2(test_book_file, temp_copy_path)
        except (OSError, PermissionError) as e:
            # Cleanup and return error
            if os.path.exists(temp_test_dir):
                try:
                    shutil.rmtree(temp_test_dir)
                except:
                    pass
            return False, f"Cannot copy files from Kindle folder: {str(e)}", test_book_file
        
        # Test 2: Verify copy succeeded
        if not os.path.exists(temp_copy_path):
            # Cleanup and return error
            if os.path.exists(temp_test_dir):
                try:
                    shutil.rmtree(temp_test_dir)
                except:
                    pass
            return False, "Copy operation completed but file does not exist", test_book_file
        
        # Test 3: Verify file size matches
        copied_size = os.path.getsize(temp_copy_path)
        if copied_size != file_size:
            # Cleanup and return error
            if os.path.exists(temp_test_dir):
                try:
                    shutil.rmtree(temp_test_dir)
                except:
                    pass
            return False, f"Copied file size mismatch (original: {file_size}, copied: {copied_size})", test_book_file
        
        # Test 4: Delete temp copy
        try:
            os.remove(temp_copy_path)
        except (OSError, PermissionError) as e:
            # Cleanup and return error
            if os.path.exists(temp_test_dir):
                try:
                    shutil.rmtree(temp_test_dir)
                except:
                    pass
            return False, f"Cannot delete temporary files: {str(e)}", test_book_file
        
        # Test 5: Cleanup temp directory
        try:
            shutil.rmtree(temp_test_dir)
        except (OSError, PermissionError) as e:
            return False, f"Cannot remove temporary directory: {str(e)}", test_book_file
        
        # All tests passed
        return True, None, test_book_file
        
    except Exception as e:
        # Cleanup on exception
        if temp_test_dir and os.path.exists(temp_test_dir):
            try:
                shutil.rmtree(temp_test_dir)
            except:
                pass
        return False, f"Unexpected error during accessibility test: {str(e)}", None


def is_cloud_synced_location(directory):
    """Detect if directory is inside a cloud sync folder"""
    dir_lower = directory.lower()
    
    cloud_patterns = {
        'onedrive': ['onedrive', '\\onedrive\\', '/onedrive/'],
        'google-drive': ['google drive', '\\google drive\\', '/google drive/', 'googledrive'],
        'dropbox': ['dropbox', '\\dropbox\\', '/dropbox/'],
        'icloud': ['icloud', '\\icloud\\', '/icloud/', 'icloud drive'],
    }
    
    for service, patterns in cloud_patterns.items():
        for pattern in patterns:
            if pattern in dir_lower:
                return True, service
    
    return False, None


def check_write_permissions(directory):
    """Check if directory is writable and not in cloud sync folder"""
    is_cloud, cloud_service = is_cloud_synced_location(directory)
    if is_cloud:
        return False, f"Cloud sync folder detected ({cloud_service}) - may cause access issues"
    
    test_dir = os.path.join(directory, ".write_test_temp_dir")
    try:
        os.makedirs(test_dir, exist_ok=True)
        os.rmdir(test_dir)
        return True, None
    except Exception as e:
        return False, str(e)


def rmtree_with_retry(path, max_retries=5, retry_delay=0.5):
    """Remove directory tree with retry logic for Windows file locking"""
    for attempt in range(max_retries):
        try:
            shutil.rmtree(path)
            return True
        except (OSError, PermissionError) as e:
            if attempt < max_retries - 1:
                time.sleep(retry_delay * (2 ** attempt))
            else:
                break
    
    # Fallback: Use Windows force-delete
    return force_delete_directory_windows(path, timeout=30)


def force_delete_directory_windows(path, timeout=30):
    """Force-delete directory using Windows rd command"""
    try:
        if not os.path.exists(path):
            return True
        
        cmd = ['cmd', '/c', 'rd', '/s', '/q', f'"{path}"']
        
        result = subprocess.run(
            cmd,
            capture_output=True,
            timeout=timeout,
            text=True,
            shell=True
        )
        
        return not os.path.exists(path)
        
    except subprocess.TimeoutExpired:
        return False
    except Exception:
        return False


def remove_file_with_retry(file_path, max_retries=3):
    """
    Delete a file with retry logic and force-delete fallback for Windows
    
    Args:
        file_path: Path to file to delete
        max_retries: Maximum number of retry attempts
        
    Returns:
        tuple: (success: bool, error_msg: str or None)
    """
    for attempt in range(max_retries):
        try:
            # First try: Normal file removal
            if os.path.exists(file_path):
                os.remove(file_path)
                
                # Verify deletion
                time.sleep(0.2)  # Brief wait for Windows filesystem
                if not os.path.exists(file_path):
                    return True, None
            else:
                return True, None  # File doesn't exist, success
                
        except PermissionError as e:
            if attempt < max_retries - 1:
                print_warn(f"File locked, retrying in 2 seconds... (attempt {attempt + 1}/{max_retries})")
                time.sleep(2)
            else:
                # Last attempt failed - try Windows force delete
                print_warn("Standard deletion failed - attempting force delete...")
                return force_delete_file_windows(file_path)
                
        except Exception as e:
            return False, f"Unexpected error during deletion: {e}"
    
    return False, "Unknown error during file deletion"


def force_delete_file_windows(file_path):
    """
    Force delete a file using Windows DEL command with force flags
    
    Args:
        file_path: Path to file to delete
        
    Returns:
        tuple: (success: bool, error_msg: str or None)
    """
    try:
        if not os.path.exists(file_path):
            return True, None
        
        # Use Windows DEL with force flag (/F = force delete read-only)
        cmd = ['cmd', '/c', 'del', '/F', '/Q', f'"{file_path}"']
        
        result = subprocess.run(
            cmd,
            capture_output=True,
            timeout=10,
            text=True,
            shell=True
        )
        
        # Verify deletion
        time.sleep(0.3)
        if not os.path.exists(file_path):
            return True, None
        else:
            return False, "File still exists after force delete"
            
    except subprocess.TimeoutExpired:
        return False, "Force delete command timed out"
    except Exception as e:
        return False, f"Force delete failed: {e}"


def is_kindle_running():
    """Check if Kindle.exe is currently running"""
    try:
        result = subprocess.run(
            ['tasklist', '/FI', 'IMAGENAME eq Kindle.exe'],
            capture_output=True,
            text=True,
            timeout=5
        )
        output_lower = result.stdout.lower()
        return 'kindle.exe' in output_lower
    except Exception:
        return True


def is_calibre_running():
    """Check if any Calibre processes are currently running"""
    try:
        # Simple approach - check for any process with "calibre" in the name
        result = subprocess.run(
            ['tasklist'],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        if result.returncode == 0:
            output_lower = result.stdout.lower()
            
            # Check for common Calibre process names
            calibre_indicators = [
                'calibre.exe',
                'calibre-parallel.exe',
                'calibre-server.exe',
                'calibredb.exe',
                'ebook-convert.exe',
                'ebook-viewer.exe',
                'ebook-edit.exe',
                'calibrePortable.exe'
            ]
            
            for indicator in calibre_indicators:
                if indicator in output_lower:
                    return True
        
        return False
    except Exception:
        return False


def wait_for_calibre_to_close(context_message="", calibre_import_enabled=True):
    """
    Wait for Calibre to close, or let user continue at their own risk.

    If Calibre is running the user is offered two choices:
        [W]ait  - poll until Calibre is closed (original behaviour)
        [C]ontinue - proceed immediately, accepting the risk

    When calibre_import_enabled is True the risk is database corruption
    (Phases 3 & 4 will be skipped by the caller).
    When calibre_import_enabled is False the risk is a permission error
    on file operations (Phases 3 & 4 are already skipped anyway).

    Args:
        context_message: Optional message explaining why Calibre needs to be closed
        calibre_import_enabled: Whether Calibre import is enabled in config

    Returns:
        True  - Calibre was closed (or was not running) - proceed normally
        False - User chose to continue with Calibre still running
    """
    if not is_calibre_running():
        return True

    print()
    print_colored("=" * 70, "red")
    print_colored(" CALIBRE IS CURRENTLY RUNNING!", "red")
    print_colored("=" * 70, "red")
    print()

    if context_message:
        print_colored(context_message, "yellow")
        print()

    if calibre_import_enabled:
        print_colored("WARNING: Continuing with Calibre open risks DATABASE CORRUPTION!", "red")
        print_colored("         If you continue, Phases 3 & 4 will be skipped to protect", "red")
        print_colored("         your Calibre library.", "red")
    else:
        print_colored("WARNING: Continuing with Calibre open may cause PERMISSION ERRORS", "yellow")
        print_colored("         on file operations. Phases 3 & 4 are already disabled,", "yellow")
        print_colored("         so the risk is limited to Phases 1 & 2.", "yellow")
    print()

    print("Options:")
    print("  [W] Wait    - Wait for Calibre to close (recommended)")
    print("  [C] Continue - Proceed at my own risk (not recommended)")
    print()

    while True:
        choice = input("Your choice (W/C) [W]: ").strip().upper()
        if choice == '' or choice == 'W':
            break
        elif choice == 'C':
            print()
            print_warn("Proceeding with Calibre still running - at user's own risk.")
            if calibre_import_enabled:
                print_warn("Phases 3 & 4 will be SKIPPED to prevent database corruption.")
            print()
            return False
        else:
            print_error("Invalid choice. Please enter W or C.")

    # User chose to wait - poll until Calibre closes
    print()
    print_step("Please close Calibre now...")
    print("(The script will automatically continue once Calibre is closed)")
    print()

    dots = 0
    while is_calibre_running():
        dots = (dots + 1) % 4
        print(f"\r  Waiting for Calibre to close{'.' * dots}   ", end='', flush=True)
        time.sleep(1)

    # Clear the waiting message
    print("\r" + " " * 50 + "\r", end='', flush=True)
    print_ok("Calibre has been closed - continuing...")
    print()

    return True


def check_calibre_installed():
    """Check if Calibre is installed"""
    try:
        result = subprocess.run(
            ['calibredb', '--version'],
            capture_output=True,
            timeout=5
        )
        return result.returncode == 0
    except:
        return False


def find_tool_by_prefix(search_dirs, prefix):
    """
    Locate a tool executable by case-insensitive filename prefix.

    Scans each directory in search_dirs (in order) for a file whose lowercased
    name starts with prefix and ends with '.exe'.  Returns the full path of the
    first match found, or None if no match exists in any of the directories.

    Args:
        search_dirs: Ordered list of directory paths to search.
        prefix:      Lowercase prefix string (e.g. 'kfxkeyextractor').

    Returns:
        str or None: Full path to the matched executable, or None.
    """
    for directory in search_dirs:
        if not os.path.isdir(directory):
            continue
        try:
            for fname in os.listdir(directory):
                if fname.lower().startswith(prefix) and fname.lower().endswith('.exe'):
                    return os.path.join(directory, fname)
        except OSError:
            continue
    return None


def check_extractor_exists(script_dir):
    """
    Locate the KFXKeyExtractor tool (any version) using case-insensitive prefix search.

    Search order:
      1. code/tools/  (canonical location for all KFX tools)
      2. script root  (project root, for backward compatibility)

    Returns:
        tuple: (found: bool, path: str or None)
    """
    tools_dir = os.path.join(script_dir, "code", "tools")
    search_dirs = [tools_dir, script_dir]
    path = find_tool_by_prefix(search_dirs, KFXKEYEXTRACTOR_PREFIX)
    return (path is not None, path)


def check_alt_extractor_exists(script_dir):
    """
    Locate the KFXArchiver tool (any version) using case-insensitive prefix search.

    Search order:
      1. code/tools/  (canonical location for all KFX tools)
      2. script root  (project root, for backward compatibility)

    Returns:
        tuple: (found: bool, path: str or None)
    """
    tools_dir = os.path.join(script_dir, "code", "tools")
    search_dirs = [tools_dir, script_dir]
    path = find_tool_by_prefix(search_dirs, KFXARCHIVER_PREFIX)
    return (path is not None, path)


def get_calibre_plugins_dir():
    """Get Calibre plugins directory path"""
    user_home = os.path.expanduser("~")
    return os.path.join(user_home, "AppData", "Roaming", "calibre", "plugins")


def check_plugin_installed(plugin_name):
    """Check if a Calibre plugin is installed"""
    plugins_dir = get_calibre_plugins_dir()
    plugin_path = os.path.join(plugins_dir, f"{plugin_name}.zip")
    return os.path.exists(plugin_path)


def get_raw_log_path(working_dir, phase_name):
    """Get path for RAW debug log file"""
    # Map phase names to numbered folders
    phase_folder_map = {
        'extraction': '01_Extraction',
        'import': '02_Import',
        'conversion': '03_Conversion'
    }
    
    folder_name = phase_folder_map.get(phase_name, f"{phase_name}_logs")
    logs_dir = os.path.join(working_dir, "Logs", folder_name)
    os.makedirs(logs_dir, exist_ok=True)

    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    return os.path.join(logs_dir, f"raw_{phase_name}_{timestamp}.log")


def append_to_raw_log(raw_log_path, command, stdout, stderr, exit_code, book_info=None):
    """Append subprocess results to RAW debug log file"""
    try:
        file_exists = os.path.exists(raw_log_path)
        
        with open(raw_log_path, 'a', encoding='utf-8') as f:
            if not file_exists:
                f.write("═" * 70 + "\n")
                phase_name = os.path.basename(os.path.dirname(raw_log_path)).replace('_logs', '').upper()
                f.write(f"RAW DEBUG LOG - {phase_name} PHASE\n")
                f.write("═" * 70 + "\n")
                f.write(f"Script Version: {SCRIPT_VERSION}\n")
                f.write(f"Log Created: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                f.write("═" * 70 + "\n\n")
            
            f.write("─" * 70 + "\n")
            f.write(f"[{datetime.now().strftime('%H:%M:%S')}]")
            if book_info:
                f.write(f" Book: {book_info}\n")
            else:
                f.write("\n")
            
            if isinstance(command, list):
                f.write(f"Command: {' '.join(command)}\n")
            else:
                f.write(f"Command: {command}\n")
            
            f.write("─" * 70 + "\n")
            f.write("STDOUT:\n")
            f.write(stdout if stdout else "(empty)\n")
            f.write("\n")
            f.write("STDERR:\n")
            f.write(stderr if stderr else "(empty)\n")
            f.write("\n")
            f.write(f"Exit Code: {exit_code}\n")
            f.write("═" * 70 + "\n\n")
            
    except Exception:
        pass


def cleanup_old_raw_logs(logs_dir, keep_count=10):
    """Cleanup old RAW log files"""
    try:
        if not os.path.exists(logs_dir):
            return
        
        raw_logs = []
        for filename in os.listdir(logs_dir):
            if filename.startswith('raw_') and filename.endswith('.log'):
                filepath = os.path.join(logs_dir, filename)
                mtime = os.path.getmtime(filepath)
                raw_logs.append((filepath, mtime))
        
        raw_logs.sort(key=lambda x: x[1], reverse=True)
        
        for filepath, _ in raw_logs[keep_count:]:
            try:
                os.remove(filepath)
            except Exception:
                pass
                
    except Exception:
        pass


# ============================================================================
# CALIBRE UTILITY FUNCTIONS
# ============================================================================

def get_library_book_count(library_path):
    """Get the total number of books in a Calibre library"""
    import sqlite3
    
    try:
        db_path = os.path.join(library_path, "metadata.db")
        conn = sqlite3.connect(db_path, timeout=5)
        cursor = conn.cursor()
        cursor.execute("SELECT COUNT(*) FROM books")
        count = cursor.fetchone()[0]
        conn.close()
        return count
    except Exception:
        return None


def verify_library_path(library_path):
    """Verify library path exists and contains metadata.db"""
    if not os.path.exists(library_path):
        return False, "Library path does not exist", None
    
    metadata_db = os.path.join(library_path, "metadata.db")
    if not os.path.exists(metadata_db):
        return False, "Not a valid Calibre library (metadata.db not found)", None
    
    book_count = get_library_book_count(library_path)
    
    return True, "", book_count


def get_last_calibre_library():
    """Read last library from Calibre's global.py.json"""
    try:
        config_file = os.path.join(
            os.path.expanduser("~"),
            "AppData", "Roaming", "calibre", "global.py.json"
        )
        
        if not os.path.exists(config_file):
            return None
        
        with open(config_file, 'r') as f:
            config = json.load(f)
        
        return config.get('library_path')
        
    except Exception:
        return None


def prompt_manual_library_path():
    """Prompt user to manually enter library path"""
    print_step("Enter Calibre Library Path")
    print("This is the folder containing metadata.db")
    print("Example: D:\\OneDrive\\Calibre-Libraries\\Temp-downloads")
    print()
    
    while True:
        library_path = input("Library path: ").strip().strip('"').strip("'")
        
        if not library_path:
            print_error("Path cannot be empty")
            continue
        
        library_path = os.path.normpath(library_path)
        
        valid, error, book_count = verify_library_path(library_path)
        
        if valid:
            print_ok(f"Library path validated: {library_path}")
            if book_count is not None:
                print(f"  Books found: {book_count}")
            else:
                print(f"  Books found: Unknown")
            print()
            return library_path
        else:
            print_error(f"Invalid library path: {error}")
            print()
            retry = input("Try again? (Y/N): ").strip().upper()
            if retry != 'Y':
                raise Exception("Library path validation cancelled")


# ============================================================================
# PHASE TRANSITION HELPER FUNCTIONS
# ============================================================================

def clear_screen_between_phases(config):
    """Clear screen if clear_screen_between_phases is enabled in config"""
    if config and config.get('clear_screen_between_phases', True):
        os.system('cls')


def display_phase_summary(phase_num, phase_name, summary_points, config, pause_seconds=5):
    """Display phase completion summary with countdown
    
    Args:
        phase_num: Phase number (1-4)
        phase_name: Name of the phase
        summary_points: List of accomplishment strings to display
        config: Configuration dict
        pause_seconds: Number of seconds for countdown (default 5)
    """
    print()
    print("=" * 70)
    print_done(f"[PHASE {phase_num}] {phase_name} - COMPLETE")
    print("=" * 70)
    print()
    print_step("Summary of accomplishments:")
    for point in summary_points:
        print(f"  ✓ {point}")
    print()
    
    # Check if pauses should be skipped
    skip_pauses = config.get('skip_phase_pauses', False) if config else False
    
    if skip_pauses:
        # Skip countdown - just display summary and continue
        print_step("Continuing to next phase...")
        print()
        return
    
    # Show countdown
    print(f"Continuing to next phase in {pause_seconds} seconds...")
    print()
    
    # Countdown
    for i in range(pause_seconds, 0, -1):
        print(f"\r  {i}...", end='', flush=True)
        time.sleep(1)
    
    print("\r" + " " * 20 + "\r", end='', flush=True)
    print()


def auto_launch_kindle(config):
    """Launch Kindle if auto_launch_kindle is enabled in config
    
    Returns True if Kindle was launched (and user closed it), False otherwise
    """
    import subprocess
    
    if not config or not config.get('auto_launch_kindle', False):
        return False
    
    print()
    print("=" * 70)
    print_step("Auto-Launch Kindle - Enabled in Configuration")
    print("=" * 70)
    print()
    
    # Check if Kindle is already running
    if is_kindle_running():
        print_warn("Kindle is already running")
        print_step("Waiting for Kindle to close before scanning books...")
        
        while is_kindle_running():
            time.sleep(1)
        
        print_ok("Kindle has closed - continuing...")
        return True
    
    # Find Kindle.exe location
    user_home = os.path.expanduser("~")
    
    appdata_kindle = os.path.join(user_home, "AppData", "Local", "Amazon", "Kindle", "application")
    appdata_exe = os.path.join(appdata_kindle, "Kindle.exe")
    
    program_files_kindle = r"C:\Program Files (x86)\Amazon\Kindle"
    program_files_exe = os.path.join(program_files_kindle, "Kindle.exe")
    
    appdata_exists = os.path.exists(appdata_exe)
    program_files_exists = os.path.exists(program_files_exe)
    
    if program_files_exists:
        kindle_dir = program_files_kindle
    elif appdata_exists:
        kindle_dir = appdata_kindle
    else:
        print_warn("Kindle.exe not found - cannot auto-launch")
        return False
    
    kindle_exe = os.path.join(kindle_dir, "Kindle.exe")
    
    if not os.path.exists(kindle_exe):
        print_warn(f"Kindle.exe not found at: {kindle_exe}")
        return False
    
    print_step("Launching Kindle.exe...")
    print(f"  Location: {kindle_exe}")
    print()
    
    # Launch Kindle
    subprocess.Popen([kindle_exe])
    
    # Wait a moment for Kindle to start
    time.sleep(2)
    
    print_ok("Kindle launched successfully")
    print_step("Waiting for Kindle to close before scanning books...")
    print("(You can close Kindle manually when ready, and the script will continue)")
    print()
    
    # Wait for Kindle to close
    while is_kindle_running():
        time.sleep(1)
    
    print_ok("Kindle has closed - continuing with key extraction...")
    return True


def prevent_kindle_auto_update():
    """
    Create/update the 'updates' file to prevent Kindle for PC from auto-updating
    Only runs if Kindle is installed
    
    This creates a simple text file in the Kindle AppData folder that prevents
    Kindle for PC from automatically checking for and installing updates.
    """
    user_home = os.path.expanduser("~")
    kindle_base = os.path.join(user_home, "AppData", "Local", "Amazon", "Kindle")
    updates_file = os.path.join(kindle_base, "updates")
    
    # Check if Kindle directory exists
    if not os.path.exists(kindle_base):
        return  # Kindle not installed, skip
    
    print_step("Configuring Kindle auto-update prevention...")
    
    try:
        # Create the updates file (no extension)
        with open(updates_file, 'w') as f:
            f.write("This file prevents Kindle for PC from auto-updating.\n")
            f.write("Created by Kindle Key Finder script.\n")
            f.write("Safe to delete if you want to allow auto-updates.\n")
        
        print_ok("Auto-update prevention file created/updated")
        print_ok(f"Location: {updates_file}")
        print("   Kindle for PC will not auto-update while this file exists")
    except Exception as e:
        print_warn(f"Could not create auto-update prevention file: {e}")
    
    print()


def display_credits_and_links():
    """
    Display final credits page with links and acknowledgments
    Can be called from multiple locations for consistency
    """
    print()
    print_banner_and_version()
    
    print("For the latest version of this script and updates, visit:")
    print_colored("https://techy-notes.com/blog/dedrm-v10-0-14-tutorial", 'cyan')
    print()
    print("Watch the video tutorial on YouTube:")
    print_colored("https://www.youtube.com/watch?v=pkii6EQEeGs", 'cyan')
    print()
    print_warn("Please subscribe to the YouTube channel!")
    print("Your support and appreciation is greatly valued.")
    print()
    print("If you'd like to show extra support this Script, consider buying me a Beer!:")
    print_colored("https://buymeacoffee.com/jadehawk", 'cyan')
    print()
    print("--------------------------------------------------")
    print_step("CREDITS")
    print("--------------------------------------------------")
    print()
    print("This script is powered by KFXKeyExtractor (by Satsuoni)")
    print("Created/Modded by: Satsuoni")
    print()
    print("KFXKeyExtractor is the CORE tool that makes this automation possible.")
    print("It extracts Kindle DRM keys from your Kindle for PC installation,")
    print("enabling the DeDRM process for your purchased ebooks.")
    print()
    print("Visit Satsuoni's GitHub profile:")
    print_colored("https://github.com/Satsuoni", 'cyan')
    print()
    print("For the DeDRM tools repository:")
    print_colored("https://github.com/Satsuoni/DeDRM_tools", 'cyan')
    print()
    print("Thank you, Satsuoni, for creating and maintaining this essential tool!")
    print()


# ============================================================================
# FAILED BOOKS TRACKING FUNCTIONS
# ============================================================================

def load_failed_books(working_dir):
    """
    Load failed books from Failed-Books.txt
    Uses working_dir which respects fallback paths
    
    Args:
        working_dir: Working directory path
    
    Returns:
        set: Set of "ASIN - Title" entries from Failed-Books.txt
    """
    failed_books_file = os.path.join(working_dir, "Failed-Books.txt")
    failed_entries = set()
    
    try:
        if os.path.exists(failed_books_file):
            with open(failed_books_file, 'r', encoding='utf-8') as f:
                for line in f:
                    entry = line.strip()
                    if entry:
                        failed_entries.add(entry)
    except Exception as e:
        print_warn(f"Failed to load failed books file: {e}")
    
    return failed_entries


def resolve_failed_book_title(asin):
    """
    Resolve book title for failed books only.
    Always attempts to fetch title regardless of fetch_book_titles config.
    
    Args:
        asin: Amazon Standard Identification Number
    
    Returns:
        str: "ASIN - Book Title" or "ASIN - Unable to resolve book's title"
    """
    try:
        # Attempt to fetch title from Amazon using fetch-ebook-metadata
        result = subprocess.run(
            ['fetch-ebook-metadata', '-I', f'asin:{asin}'],
            capture_output=True,
            text=True,
            timeout=10,
            encoding='utf-8',
            errors='replace'
        )
        
        if result.returncode == 0 and result.stdout:
            # Parse output for title
            for line in result.stdout.split('\n'):
                if line.strip().startswith('Title'):
                    # Format: "Title               : The Sunken: A Dark Steampunk Fantasy"
                    parts = line.split(':', 1)
                    if len(parts) == 2:
                        title = parts[1].strip()
                        if title:
                            return f"{asin} - {title}"
        
        # If fetch failed or no title found, return generic message
        return f"{asin} - Unable to resolve book's title"
        
    except Exception:
        return f"{asin} - Unable to resolve book's title"


def remove_from_failed_books(working_dir, asin):
    """
    Remove a book from Failed-Books.txt by ASIN (used when alt retry succeeds).
    Rewrites the file without the matching entry.

    Args:
        working_dir: Working directory path
        asin: Amazon Standard Identification Number to remove
    """
    failed_books_file = os.path.join(working_dir, "Failed-Books.txt")

    if not os.path.exists(failed_books_file):
        return

    try:
        with open(failed_books_file, 'r', encoding='utf-8') as f:
            lines = f.readlines()

        # Keep lines that do NOT start with this ASIN
        filtered = [ln for ln in lines if not ln.strip().startswith(f"{asin} -") and ln.strip() != asin]

        if len(filtered) == len(lines):
            return  # Nothing changed, skip rewrite

        with open(failed_books_file, 'w', encoding='utf-8') as f:
            f.writelines(filtered)

    except Exception as e:
        print_warn(f"Failed to update Failed-Books.txt after alt retry: {e}")


def append_to_failed_books(working_dir, title, asin):
    """
    Append book to Failed-Books.txt (no duplicates)
    Format: "ASIN - Book Title"
    Uses working_dir which respects fallback paths
    
    ALWAYS attempts to resolve title for failed books
    
    Args:
        working_dir: Working directory path
        title: Book title (or ASIN if title not available)
        asin: Amazon Standard Identification Number
    """
    failed_books_file = os.path.join(working_dir, "Failed-Books.txt")
    
    try:
        # Ensure directory exists
        os.makedirs(os.path.dirname(failed_books_file), exist_ok=True)
        
        # Load existing entries to check for duplicates
        existing_entries = load_failed_books(working_dir)
        
        # Check if title is actually an ASIN (no title was fetched)
        # This happens when fetch_book_titles is disabled
        if title == asin:
            # Always resolve title for failed books
            entry = resolve_failed_book_title(asin)
        else:
            # Title was already fetched, reformat as "ASIN - Title"
            entry = f"{asin} - {title}"
        
        # Only append if not already in file
        if entry not in existing_entries:
            with open(failed_books_file, 'a', encoding='utf-8') as f:
                f.write(f"{entry}\n")
    except Exception as e:
        print_warn(f"Failed to update failed books file: {e}")
