#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Kindle Key Finder - Python Edition Wrapper
Replicates the exact DeDRM plugin logic for key extraction and JSON generation
No external dependencies - uses the same methods as the plugin
"""

# Script Version
SCRIPT_VERSION = "2025.11.22.JH"

# Unified Configuration File
CONFIG_FILE = "key_finder_config.json"

import os
import sys
import json
import subprocess
import shutil
import time
import threading
import msvcrt
import platform
from datetime import datetime

def print_colored(message: str, color: str) -> None:
    """Print colored messages"""
    colors = {
        'cyan': '\033[96m',
        'green': '\033[92m',
        'yellow': '\033[93m',
        'red': '\033[91m',
        'magenta': '\033[95m',
        'end': '\033[0m'
    }
    print(f"{colors.get(color, '')}{message}{colors['end']}")

def print_step(message: str) -> None:
    print_colored(f"[*] {message}", 'cyan')

def print_ok(message: str) -> None:
    print_colored(f"[OK] {message}", 'green')

def print_warn(message: str) -> None:
    print_colored(f"[!] {message}", 'yellow')

def print_error(message: str) -> None:
    print_colored(f"[ERROR] {message}", 'red')

def print_done(message: str) -> None:
    print_colored(f"[DONE] {message}", 'magenta')

def parse_version(version_str):
    """
    Parse version string into comparable components.
    Format: YYYY.MM.DD[.build]JH
    
    Args:
        version_str: Version string (e.g., "2025.11.11.JH" or "2025.11.11.1.JH")
    
    Returns:
        tuple: (year, month, day, build) or None if parsing fails
        - build defaults to 0 if not present
    
    Example:
        "2025.11.11.JH" -> (2025, 11, 11, 0)
        "2025.11.11.1.JH" -> (2025, 11, 11, 1)
    """
    try:
        # Remove .JH suffix if present
        version_clean = version_str.replace('.JH', '').replace('JH', '')
        
        # Split into parts
        parts = version_clean.split('.')
        
        if len(parts) < 3:
            return None
        
        year = int(parts[0])
        month = int(parts[1])
        day = int(parts[2])
        build = int(parts[3]) if len(parts) >= 4 else 0
        
        # Validate ranges
        if not (2000 <= year <= 2100 and 1 <= month <= 12 and 1 <= day <= 31):
            return None
        
        return (year, month, day, build)
        
    except (ValueError, IndexError):
        return None

def compare_versions(version1, version2):
    """
    Compare two version strings chronologically.
    
    Args:
        version1: First version string
        version2: Second version string
    
    Returns:
        int: -1 if version1 < version2
             0 if version1 == version2
             1 if version1 > version2
             None if either version cannot be parsed
    """
    v1 = parse_version(version1)
    v2 = parse_version(version2)
    
    if v1 is None or v2 is None:
        return None
    
    # Compare tuples directly (year, month, day, build)
    if v1 < v2:
        return -1
    elif v1 > v2:
        return 1
    else:
        return 0

def check_latest_version():
    """
    Check for latest version from GitHub repository releases
    Returns: (latest_version: str or None, error_message: str or None)
    """
    try:
        import urllib.request
        import json
        
        # GitHub API URL for latest release
        url = "https://api.github.com/repos/jadehawk/Kindle_Key_Finder/releases/latest"
        
        # Set timeout to 5 seconds
        req = urllib.request.Request(url, headers={'User-Agent': 'Mozilla/5.0'})
        
        with urllib.request.urlopen(req, timeout=5) as response:
            data = json.loads(response.read().decode('utf-8'))
            
            # Get tag_name from release (e.g., "2025.11.11.JH" or "2025.11.11.1.JH")
            tag_name = data.get('tag_name', '')
            
            if tag_name:
                # Remove 'v' prefix if present (e.g., "v2025.11.11.JH" -> "2025.11.11.JH")
                version = tag_name.lstrip('v')
                return version, None
            else:
                return None, "No tag_name found in latest release"
                
    except Exception as e:
        # Silently fail - don't disrupt script execution
        return None, str(e)

def print_banner_and_version():
    """Print ASCII banner and script version"""
    print_colored(r" _____         _                   _   _       _                  ____                ", 'green')
    print_colored(r"|_   _|__  ___| |__  _   _        | \ | | ___ | |_ ___  ___      / ___|___  _ __ ___  ", 'green')
    print_colored(r"  | |/ _ \/ __| '_ \| | | | _____ |  \| |/ _ \| __/ _ \/ __|    | |   / _ \| '_ ` _ \ ", 'green')
    print_colored(r"  | |  __/ (__| | | | |_| ||_____|| |\  | (_) | ||  __/\__ \  _ | |__| (_) | | | | | |", 'green')
    print_colored(r"  |_|\___|\___|_| |_|\__, |       |_| \_|\___/ \__\___||___/ (_) \____\___/|_| |_| |_|", 'green')
    print_colored(r"                     |___/                                                             ", 'green')
    print()
    print_step("Kindle Key Finder - Python Edition Wrapper")
    print(f"Script Version: {SCRIPT_VERSION}")
    print()
    
    # Check for latest version from GitHub
    print_step("Checking for updates...")
    latest_version, error = check_latest_version()
    
    if latest_version:
        comparison = compare_versions(SCRIPT_VERSION, latest_version)
        
        if comparison is None:
            # Parsing failed - fall back to string comparison
            print_colored("  (Unable to parse version - check manually)", 'cyan')
            print_colored("  Visit https://techy-notes.com for the latest version", 'cyan')
            print()
        elif comparison < 0:
            # Local version is older than GitHub version
            print()
            print_colored("=" * 70, 'yellow')
            print_colored(f"  ⚠   NEW VERSION AVAILABLE: {latest_version}", 'yellow')
            print_colored(f"  Current version: {SCRIPT_VERSION}", 'yellow')
            print_colored("  Visit https://techy-notes.com for the latest version", 'yellow')
            print_colored("=" * 70, 'yellow')
            print()
        elif comparison > 0:
            # Local version is NEWER than GitHub version (dev/testing scenario)
            print_colored(f"  ℹ   You are running a development version ({SCRIPT_VERSION})", 'cyan')
            print_colored(f"  Latest stable release: {latest_version}", 'cyan')
            print()
        else:
            # Versions are equal
            print_colored("  ✓   You are running the latest version", 'green')
            print()
    else:
        # Connection failed - show fallback message with URL
        print_colored("  (Unable to check for updates - connection failed)", 'cyan')
        print_colored("  Visit https://techy-notes.com for the latest version", 'cyan')
        print()

def display_credits_and_links():
    """
    Display final credits page with links and acknowledgments
    Can be called from multiple locations for consistency
    """
    os.system('cls')
    print()
    print_banner_and_version()
    
    # Check for latest version (self-contained)
    latest_version, error = check_latest_version()
    
    if latest_version:
        comparison = compare_versions(SCRIPT_VERSION, latest_version)
        
        if comparison is None:
            # Parsing failed - fall back to string comparison
            print_colored("  (Unable to parse version - check manually)", 'cyan')
            print_colored("  Visit https://techy-notes.com for the latest version", 'cyan')
            print()
        elif comparison < 0:
            # Local version is older than GitHub version
            print()
            print_colored("=" * 70, 'yellow')
            print_colored(f"  ⚠   NEW VERSION AVAILABLE: {latest_version}", 'yellow')
            print_colored(f"  Current version: {SCRIPT_VERSION}", 'yellow')
            print_colored("  Visit https://techy-notes.com for the latest version", 'yellow')
            print_colored("=" * 70, 'yellow')
            print()
        elif comparison > 0:
            # Local version is NEWER than GitHub version (dev/testing scenario)
            print_colored(f"  ℹ   You are running a development version ({SCRIPT_VERSION})", 'cyan')
            print_colored(f"  Latest stable release: {latest_version}", 'cyan')
            print()
        else:
            # Versions are equal
            print_colored("  ✓   You are running the latest version", 'green')
            print()
    else:
        # Connection failed - show fallback message with URL
        print_colored("  (Unable to check for updates - connection failed)", 'cyan')
        print_colored("  Visit https://techy-notes.com for the latest version", 'cyan')
        print()
    
    print("For the latest version of this script and updates, visit:")
    print_colored("https://techy-notes.com/blog/dedrm-v10-0-14-tutorial", 'cyan')
    print()
    print("Watch the video tutorial on YouTube:")
    print_colored("https://www.youtube.com/watch?v=pkii6EQEeGs", 'cyan')
    print()
    print_warn("Please subscribe to the YouTube channel!")
    print("Your support and appreciation is greatly valued.")
    print()
    print("If you'd like to show extra support this Script, consider buying me a Beer!:")
    print_colored("https://buymeacoffee.com/jadehawk", 'cyan')
    print()
    print("--------------------------------------------------")
    print_step("CREDITS")
    print("--------------------------------------------------")
    print()
    print("This script is powered by KFXKeyExtractor28.exe")
    print("Created/Modded by: Satsuoni")
    print()
    print("KFXKeyExtractor is the CORE tool that makes this automation possible.")
    print("It extracts Kindle DRM keys from your Kindle for PC installation,")
    print("enabling the DeDRM process for your purchased ebooks.")
    print()
    print("Visit Satsuoni's GitHub profile:")
    print_colored("https://github.com/Satsuoni", 'cyan')
    print()
    print("For the DeDRM tools repository:")
    print_colored("https://github.com/Satsuoni/DeDRM_tools", 'cyan')
    print()
    print("Thank you, Satsuoni, for creating and maintaining this essential tool!")
    print()

def display_phase_banner(phase_num, phase_name):
    """
    Display a prominent phase banner with decorative borders
    """
    print()
    print_colored("═" * 70, 'cyan')
    print_colored(f"║{f'PHASE {phase_num}':^68}║", 'cyan')
    print_colored(f"║{phase_name.upper():^68}║", 'cyan')
    print_colored(f"║{f'Script Version: {SCRIPT_VERSION}':^68}║", 'cyan')
    print_colored("═" * 70, 'cyan')
    print()

# ============================================================================
# PHASE SUMMARY FUNCTION
# ============================================================================

def display_phase_summary(phase_num, phase_name, summary_points, pause_seconds=5):
    """
    Display phase completion summary with countdown
    Allows user to press any key to skip countdown
    Respects skip_phase_pauses configuration flag
    """
    print()
    print("=" * 70)
    print_done(f"[PHASE {phase_num}] {phase_name} - COMPLETE")
    print("=" * 70)
    print()
    print_step("Summary of accomplishments:")
    for point in summary_points:
        print(f"  ✓ {point}")
    print()
    
    # Check if pauses should be skipped
    config = load_config()
    skip_pauses = config.get('skip_phase_pauses', False) if config else False
    
    if skip_pauses:
        # Skip countdown - just display summary and continue
        print_step("Continuing to next phase...")
        print()
        return
    
    # Show countdown with skip capability
    print(f"Continuing to next phase in {pause_seconds} seconds...")
    print("(Press any key to skip countdown)")
    print()
    
    # Countdown with skip capability
    for i in range(pause_seconds, 0, -1):
        if msvcrt.kbhit():
            msvcrt.getch()  # Consume the keypress
            print("\r" + " " * 50 + "\r", end='', flush=True)
            print_step("Countdown skipped by user")
            print()
            return
        print(f"\r  {i}...", end='', flush=True)
        time.sleep(1)
    
    print("\r" + " " * 20 + "\r", end='', flush=True)
    print()

def obfuscate_sensitive(text):
    """
    Obfuscate sensitive strings by showing first 2 and last 2 characters
    Example: 'Pd545vnnr861r5P0Pt6ttP7nP6tA6b57Pt7fS1rh' -> 'Pd**********************rh'
    """
    if len(text) <= 4:
        return text  # Too short to obfuscate meaningfully
    return text[:2] + '*' * (len(text) - 4) + text[-2:]

def filter_sensitive_output(text, hide_sensitive=False):
    """
    Filter and obfuscate sensitive information in output text if hide_sensitive is enabled
    Also always suppresses harmless Qt and Fontconfig error messages
    """
    import re
    lines = text.split('\n')
    filtered_lines = []
    
    for line in lines:
        # Always skip Qt and Fontconfig error messages (regardless of hide_sensitive)
        if 'QObject::startTimer: Timers can only be used with threads started with QThread' in line:
            continue
        if 'Fontconfig error: Cannot load default config file' in line:
            continue
        
        # If obfuscation is disabled, keep the line as-is
        if not hide_sensitive:
            filtered_lines.append(line)
            continue
        
        filtered_line = line
        
        # Obfuscate DSN values
        if line.startswith('DSN '):
            dsn_value = line.replace('DSN ', '').strip()
            if dsn_value:
                filtered_line = f"DSN {obfuscate_sensitive(dsn_value)}"
        
        # Obfuscate Tokens
        elif line.startswith('Tokens '):
            tokens_value = line.replace('Tokens ', '').strip()
            if tokens_value:
                # Handle comma-separated tokens
                if ',' in tokens_value:
                    token_parts = tokens_value.split(',')
                    obfuscated_parts = [obfuscate_sensitive(part.strip()) for part in token_parts]
                    filtered_line = f"Tokens {','.join(obfuscated_parts)}"
                else:
                    filtered_line = f"Tokens {obfuscate_sensitive(tokens_value)}"
        
        # Obfuscate DRM key with UUID and secret key
        elif 'amzn1.drm-key.v1.' in line and '$secret_key:' in line:
            # Pattern: amzn1.drm-key.v1.UUID$secret_key:SECRET
            match = re.search(r'(amzn1\.drm-key\.v1\.)([a-f0-9\-]+)(\$secret_key:)([a-f0-9]+)', line)
            if match:
                prefix = match.group(1)  # amzn1.drm-key.v1.
                uuid = match.group(2)     # UUID
                middle = match.group(3)   # $secret_key:
                secret = match.group(4)   # secret value
                
                obfuscated_uuid = obfuscate_sensitive(uuid)
                obfuscated_secret = obfuscate_sensitive(secret)
                
                original = f"{prefix}{uuid}{middle}{secret}"
                replacement = f"{prefix}{obfuscated_uuid}{middle}{obfuscated_secret}"
                filtered_line = line.replace(original, replacement)
        
        # Obfuscate secret keys (fallback for lines without UUID)
        elif '$secret_key:' in line:
            parts = line.split('$secret_key:')
            if len(parts) == 2:
                prefix = parts[0]
                secret = parts[1].strip()
                if secret:
                    filtered_line = f"{prefix}$secret_key:{obfuscate_sensitive(secret)}"
        
        # Obfuscate "Opened book with secret:" (handles both hex and base64)
        elif 'Opened book with secret:' in line:
            match = re.search(r'(Opened book with secret:\s*)([A-Za-z0-9+/=]+)', line)
            if match:
                prefix = match.group(1)
                secret = match.group(2)
                obfuscated = obfuscate_sensitive(secret)
                filtered_line = line.replace(f"{prefix}{secret}", f"{prefix}{obfuscated}")
        
        # Obfuscate "Opened book with reused secret:" (handles both hex and base64)
        elif 'Opened book with reused secret:' in line:
            match = re.search(r'(Opened book with reused secret:\s*)([A-Za-z0-9+/=]+)', line)
            if match:
                prefix = match.group(1)
                secret = match.group(2)
                obfuscated = obfuscate_sensitive(secret)
                filtered_line = line.replace(f"{prefix}{secret}", f"{prefix}{obfuscated}")
        
        # Obfuscate "Working secret:" (handles both hex and base64)
        elif 'Working secret:' in line:
            match = re.search(r'(Working secret:\s*")([A-Za-z0-9+/=]+)(")', line)
            if match:
                prefix = match.group(1)
                secret = match.group(2)
                suffix = match.group(3)
                obfuscated = obfuscate_sensitive(secret)
                filtered_line = line.replace(f"{prefix}{secret}{suffix}", f"{prefix}{obfuscated}{suffix}")
        
        # Obfuscate device_serial_number in JSON format
        elif '"device_serial_number":"' in line:
            match = re.search(r'"device_serial_number":"([^"]+)"', line)
            if match:
                serial = match.group(1)
                obfuscated = obfuscate_sensitive(serial)
                filtered_line = line.replace(f'"{serial}"', f'"{obfuscated}"')
        
        filtered_lines.append(filtered_line)
    
    return '\n'.join(filtered_lines)

def display_progress_timer(message, timeout_seconds, timer_stopped_event):
    """
    Reusable progress timer display function
    Shows elapsed time vs total timeout in [MM:SS / MM:SS] format
    
    Args:
        message: Text to display (e.g., "Converting to EPUB", "Step 1/2 (AZW3→MOBI)")
        timeout_seconds: Total timeout duration in seconds
        timer_stopped_event: threading.Event() to monitor for stop signal
    """
    start_time = time.time()
    while not timer_stopped_event.is_set():
        elapsed = int(time.time() - start_time)
        minutes, seconds = divmod(elapsed, 60)
        timeout_mins, timeout_secs = divmod(timeout_seconds, 60)
        print(f"\r  {message}.. [ {minutes:02d}:{seconds:02d} / {timeout_mins:02d}:{timeout_secs:02d} ]", 
              end='', flush=True)
        time.sleep(1)

def validate_kindle_path(path: str) -> dict:
    """
    Validate Kindle content path for books and database
    
    Scans the directory for:
    - *.azw files (Kindle book format)
    - book_asset.db (Kindle for PC database)
    
    Args:
        path: Directory path to validate
    
    Returns:
        dict: {
            "books_found": int,      # Count of *.azw files
            "db_found": bool         # book_asset.db present
        }
    """
    result = {
        "books_found": 0,
        "db_found": False
    }
    
    try:
        if not os.path.exists(path):
            return result
        
        # Search recursively for both books and database
        for root, dirs, files in os.walk(path):
            for file in files:
                # Check for .azw files
                if file.lower().endswith('.azw'):
                    result["books_found"] += 1
                
                # Check for book_asset.db
                if file.lower() == "book_asset.db":
                    result["db_found"] = True
        
        return result
        
    except Exception as e:
        print_warn(f"Error validating path: {e}")
        return result

def get_kindle_content_path(default_path, use_timer=False):
    """
    Prompt user to confirm or modify the Kindle content directory path
    Validates path immediately after selection to prevent Phase 1 failures
    Handles Windows-specific path scenarios with optional auto-proceed timer
    
    Args:
        default_path: Default Kindle content directory path
        use_timer: If True, auto-proceed after 5 seconds. If False, wait for user input.
    """
    import threading
    import msvcrt
    
    print_step("Kindle-4-PC Book's Path Configuration")
    print("--------------------------------------------------")
    print(f"Default path: {default_path}")
    print()
    
    if use_timer:
        print_warn("Press Enter to accept default immediately, or ANY KEY to Stop Timer")
        print("(Auto-proceeding with default in 5 seconds if no input...)")
    else:
        print("Press Enter to accept default, or start typing your NEW custom path below:")
    print()
    
    # Shared state for timer and input
    timer_cancelled = threading.Event()
    user_input_started = threading.Event()
    countdown_active = use_timer  # Only activate countdown if timer is enabled
    
    def countdown_timer():
        nonlocal countdown_active
        if not use_timer:
            return
        for i in range(5, 0, -1):
            if timer_cancelled.is_set() or user_input_started.is_set():
                countdown_active = False
                return
            print(f"\rCountdown: {i} seconds... ", end='', flush=True)
            time.sleep(1)
        countdown_active = False
        if not user_input_started.is_set():
            print("\r" + " " * 50 + "\r", end='', flush=True)  # Clear countdown line
            print_ok("Auto-proceeding with default path")
            timer_cancelled.set()
    
    # Start countdown timer only if enabled
    if use_timer:
        timer_thread = threading.Thread(target=countdown_timer, daemon=True)
        timer_thread.start()
    
    # Wait for user input or timer expiry
    user_input = ""
    input_buffer = []
    
    while countdown_active or user_input_started.is_set() or not use_timer:
        if msvcrt.kbhit():
            char = msvcrt.getwche()
            
            if not user_input_started.is_set():
                # First keypress - cancel timer if active
                user_input_started.set()
                if use_timer:
                    timer_cancelled.set()
                    print("\r" + " " * 50 + "\r", end='', flush=True)  # Clear countdown
                
                # Check if it's Enter key
                if char == '\r':
                    print()
                    user_input = ""
                    break
                else:
                    print("> ", end='', flush=True)
                    print(char, end='', flush=True)
                    input_buffer.append(char)
            else:
                # Continue collecting input
                if char == '\r':
                    print()
                    user_input = ''.join(input_buffer)
                    break
                elif char == '\b':  # Backspace
                    if input_buffer:
                        input_buffer.pop()
                        print(' \b', end='', flush=True)
                else:
                    input_buffer.append(char)
        elif not countdown_active and not user_input_started.is_set():
            # Timer expired (if was enabled), no input
            if use_timer:
                break
        else:
            time.sleep(0.05)
    
    user_input = user_input.strip()
    
    # If user pressed Enter without typing, use default
    if not user_input:
        content_path = default_path
    else:
        # Clean up the input path
        # Remove quotation marks (both single and double)
        content_path = user_input.strip('"').strip("'")
        
        # Expand environment variables like %USERPROFILE%
        content_path = os.path.expandvars(content_path)
        
        # Normalize path separators for Windows (convert / to \)
        content_path = os.path.normpath(content_path)
    
    # Validate path exists
    if not os.path.exists(content_path):
        print_error(f"Path does not exist: {content_path}")
        print()
        # Ask again or exit
        retry = input("Would you like to try again? (y/n): ").lower()
        if retry == 'y':
            return get_kindle_content_path(default_path)
        else:
            raise FileNotFoundError(f"Kindle content directory not found: {content_path}")

    os.system('cls')
    # Path exists - now validate for books and database
    print_step("Validating Kindle content path...")
    validation_result = validate_kindle_path(content_path)
    
    books_found = validation_result["books_found"]
    db_found = validation_result["db_found"]
    
    # Display validation results
    print()
    print("=" * 70)
    print_step("Path Validation Results:")
    print("=" * 70)
    print()
    
    if books_found > 0 and db_found:
        # Scenario A: Both books and database found (ideal)
        print_ok(f"✓ Found {books_found} book(s) (*.azw files)")
        print_ok(f"✓ Found Kindle database (book_asset.db)")
        print()
        print_colored("  This appears to be a valid Kindle for PC download folder.", 'green')
    elif books_found > 0 and not db_found:
        # Scenario B: Books found but no database (custom DeDRM folder)
        print_ok(f"✓ Found {books_found} book(s) (*.azw files)")
        print_warn(f"✗ Kindle database (book_asset.db) NOT found")
        print()
        print_colored("  This appears to be a custom folder for DeDRM purposes.", 'yellow')
        print_colored("  (Books present but not the standard Kindle for PC folder)", 'yellow')
    elif books_found == 0 and db_found:
        # Database found but no books
        print_warn(f"✗ No books (*.azw files) found")
        print_ok(f"✓ Found Kindle database (book_asset.db)")
        print()
        print_colored("  This is a Kindle for PC folder but contains no downloaded books.", 'yellow')
    else:
        # Neither books nor database found
        print_error(f"✗ No books (*.azw files) found")
        print_error(f"✗ Kindle database (book_asset.db) NOT found")
        print()
        print_colored("  This folder appears empty or is not a Kindle content folder.", 'red')
    
    print()
    print_colored("─" * 70, 'cyan')
    print_warn("IMPORTANT: Phase 1 (Key Extraction) Requirements")
    print_colored("─" * 70, 'cyan')
    print()
    
    if books_found == 0:
        print_error("⚠  Choosing this path will cause Phase 1 to fail with:")
        print_error("   'No books found in content directory!'")
        print()
        print("   You MUST download your books into this folder for the script to work.")
        print("   Or Re-enter and Choose a different path")
        print_warn(f"   Path {default_path}")
        print()
    else:
        print_ok(f"✓ Path contains {books_found} book(s) - Phase 1 should proceed normally")
        print()
    
    # User confirmation
    print("Options:")
    print("  [C] Confirm - Use this path and continue")
    print("  [R] Re-enter - Choose a different path")
    print("  [Q] Quit - Exit script")
    print()
    
    while True:
        choice = input("Your choice (C/R/Q): ").strip().upper()
        if choice == 'C':
            print()
            print_ok(f"Using path: {content_path}")
            print()
            return content_path
        elif choice == 'R':
            print()
            print_step("Re-entering path...")
            print()
            return get_kindle_content_path(default_path)
        elif choice == 'Q':
            print()
            print_warn("Script cancelled by user")
            sys.exit(0)
        else:
            print_error("Invalid choice. Please enter C, R, or Q.")

def cleanup_temp_kindle():
    """
    Check for and cleanup any leftover temporary Kindle installation
    from previous failed runs
    """
    user_home = os.path.expanduser("~")
    temp_kindle_dir = os.path.join(user_home, "AppData", "Local", "Amazon", "Kindle", "application")
    temp_marker = os.path.join(temp_kindle_dir, "TEMP.txt")
    
    if os.path.exists(temp_marker):
        print_warn("Found leftover temporary Kindle installation from previous run")
        print_step("Cleaning up...")
        try:
            shutil.rmtree(temp_kindle_dir)
            print_ok("Cleanup completed successfully")
        except Exception as e:
            print_error(f"Failed to cleanup temporary files: {e}")
            print_warn("You may need to manually delete: " + temp_kindle_dir)
        print()

def force_delete_directory_windows(path, timeout=30):
    """
    Force-delete directory using Windows rd command (bypasses file locks)
    This is more aggressive than shutil.rmtree() and won't hang on OneDrive locks
    
    Args:
        path: Directory path to remove
        timeout: Maximum time in seconds to wait for deletion (default: 30)
    
    Returns:
        bool: True if successful, False if failed or timed out
    """
    try:
        if not os.path.exists(path):
            return True  # Already deleted
        
        # Use Windows rd command with /s (recursive) and /q (quiet) flags
        # This forces deletion even with file locks that would block shutil.rmtree()
        cmd = ['cmd', '/c', 'rd', '/s', '/q', f'"{path}"']
        
        result = subprocess.run(
            cmd,
            capture_output=True,
            timeout=timeout,
            text=True,
            shell=True  # Required for proper path handling with quotes
        )
        
        # Verify deletion
        if not os.path.exists(path):
            return True
        
        # If directory still exists, try aggressive recursive deletion of subdirectories first
        # This handles OneDrive's nested empty folder locking
        try:
            for root, dirs, files in os.walk(path, topdown=False):
                # Delete files first
                for name in files:
                    try:
                        file_path = os.path.join(root, name)
                        os.chmod(file_path, 0o777)  # Force permissions
                        os.remove(file_path)
                    except:
                        pass
                
                # Delete directories
                for name in dirs:
                    try:
                        dir_path = os.path.join(root, name)
                        os.chmod(dir_path, 0o777)  # Force permissions
                        os.rmdir(dir_path)
                    except:
                        pass
            
            # Finally try to remove the root directory
            os.chmod(path, 0o777)
            os.rmdir(path)
            
            return not os.path.exists(path)
        except:
            pass
        
        # Last resort: Try rd command one more time after manual cleanup
        result = subprocess.run(
            cmd,
            capture_output=True,
            timeout=timeout,
            text=True,
            shell=True
        )
        
        return not os.path.exists(path)
        
    except subprocess.TimeoutExpired:
        return False
    except Exception:
        return False

def rmtree_with_retry(path, max_retries=5, retry_delay=0.5):
    """
    Remove directory tree with retry logic for Windows file locking
    Falls back to Windows rd command if shutil.rmtree() hangs
    
    Windows often holds file handles open briefly after processes complete,
    causing PermissionError or Access Denied errors. This function retries
    the deletion with exponential backoff, then uses force-delete on failure.
    
    Args:
        path: Directory path to remove
        max_retries: Maximum number of retry attempts (default: 5)
        retry_delay: Initial delay between retries in seconds (default: 0.5)
    
    Returns:
        bool: True if successful, False if all retries failed
    """
    # First try: Standard shutil.rmtree with retries (fast path)
    for attempt in range(max_retries):
        try:
            shutil.rmtree(path)
            return True
        except (OSError, PermissionError) as e:
            if attempt < max_retries - 1:
                # Wait with exponential backoff before retry
                time.sleep(retry_delay * (2 ** attempt))
            else:
                # All shutil retries failed
                break
    
    # Fallback: Use Windows force-delete (handles OneDrive locks)
    return force_delete_directory_windows(path, timeout=30)

def cleanup_temp_extraction(silent=False, working_dir=None):
    """
    Check for and cleanup any leftover temp_extraction folder
    from previous failed runs
    
    Args:
        silent: If True, don't print messages (for use during extraction)
        working_dir: Directory to check for temp_extraction (respects fallback paths)
    """
    if working_dir is None:
        # Determine working_dir if not provided
        script_dir = os.path.dirname(os.path.abspath(__file__))
        user_home = os.path.expanduser("~")
        can_write, _ = check_write_permissions(script_dir)
        working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    temp_extraction_dir = os.path.join(working_dir, "temp_extraction")
    
    if os.path.exists(temp_extraction_dir):
        if not silent:
            print_warn("Found leftover temp_extraction folder from previous run")
            print_step("Cleaning up...")
        
        success = rmtree_with_retry(temp_extraction_dir)
        
        if success:
            if not silent:
                print_ok("Cleanup completed successfully")
        else:
            if not silent:
                print_error("Failed to cleanup temp_extraction folder after multiple retries")
                print_warn("You may need to manually delete: " + temp_extraction_dir)
        
        if not silent:
            print()

def cleanup_temp_staging(silent=False, working_dir=None):
    """
    Check for and cleanup any leftover temp_kindle_content staging folder
    from previous failed runs
    
    Args:
        silent: If True, don't print messages
        working_dir: Directory to check for staging (respects fallback paths)
    """
    if working_dir is None:
        # Determine working_dir if not provided
        script_dir = os.path.dirname(os.path.abspath(__file__))
        user_home = os.path.expanduser("~")
        can_write, _ = check_write_permissions(script_dir)
        working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    staging_dir = os.path.join(working_dir, "temp_kindle_content")
    
    if os.path.exists(staging_dir):
        if not silent:
            print_warn("Found leftover staging folder from previous run")
            print_step("Cleaning up...")
        
        success = rmtree_with_retry(staging_dir)
        
        if success:
            if not silent:
                print_ok("Cleanup completed successfully")
        else:
            if not silent:
                print_error("Failed to cleanup staging folder after multiple retries")
                print_warn("You may need to manually delete: " + staging_dir)
        
        if not silent:
            print()

def find_kindle_exe():
    """
    Search for Kindle.exe in the two known installation locations
    Returns tuple: (kindle_dir, is_temp_copy_needed)
    """
    user_home = os.path.expanduser("~")
    
    # Location 1: AppData (default installation)
    appdata_kindle = os.path.join(user_home, "AppData", "Local", "Amazon", "Kindle", "application")
    appdata_exe = os.path.join(appdata_kindle, "Kindle.exe")
    temp_marker = os.path.join(appdata_kindle, "TEMP.txt")
    
    # Location 2: Program Files (x86)
    program_files_kindle = r"C:\Program Files (x86)\Amazon\Kindle"
    program_files_exe = os.path.join(program_files_kindle, "Kindle.exe")
    
    print_step("Searching for Kindle installation...")
    print("--------------------------------------------------")
    
    # Check both locations
    appdata_exists = os.path.exists(appdata_exe)
    program_files_exists = os.path.exists(program_files_exe)
    
    # CASE 1: Both locations have Kindle.exe
    if appdata_exists and program_files_exists:
        print_warn("Found Kindle.exe in BOTH locations:")
        print(f"  1. {appdata_kindle}")
        print(f"  2. {program_files_kindle}")
        print()
        print_error("CONFLICT DETECTED!")
        print("This suggests Kindle is installed in Global mode (Program Files)")
        print("but there's also a copy in AppData (possibly leftover or temp copy).")
        print()
        print_warn("Recommended action: Delete AppData copy and use Program Files installation")
        print()
        print("Options:")
        print("  [C] Continue - Delete AppData copy and proceed with Program Files installation")
        print("  [Q] Quit - Exit script and resolve manually")
        print()
        
        while True:
            choice = input("Your choice (C/Q): ").strip().upper()
            if choice == 'C':
                print()
                print_step("Deleting AppData Kindle copy...")
                try:
                    shutil.rmtree(appdata_kindle)
                    print_ok("AppData copy deleted successfully")
                    print()
                    # Now proceed with Program Files installation
                    print_ok(f"Using Kindle at: {program_files_kindle}")
                    print_warn("KFXKeyExtractor requires Kindle in AppData location")
                    print()
                    print_step("Solution: Temporary copy will be created")
                    print("  - Source: " + program_files_kindle)
                    print("  - Destination: " + appdata_kindle)
                    print("  - This copy will be automatically deleted after extraction")
                    print("--------------------------------------------------")
                    print()
                    return program_files_kindle, True
                except Exception as e:
                    print_error(f"Failed to delete AppData copy: {e}")
                    print_error("Please manually delete the folder and try again")
                    print("--------------------------------------------------")
                    print()
                    return None, False
            elif choice == 'Q':
                print()
                print_warn("Script cancelled by user")
                print("Please manually resolve the dual installation before running again")
                print("--------------------------------------------------")
                print()
                return None, False
            else:
                print_error("Invalid choice. Please enter C or Q.")
    
    # CASE 2: Only AppData location has Kindle.exe
    elif appdata_exists:
        # Make sure it's not a temp copy (shouldn't have TEMP.txt if real installation)
        if not os.path.exists(temp_marker):
            print_ok(f"Found Kindle at: {appdata_kindle}")
            print_ok("Using existing installation (no temporary copy needed)")
            print("--------------------------------------------------")
            print()
            return appdata_kindle, False
        else:
            # Has TEMP.txt marker - this is a leftover temp copy
            print_warn(f"Found Kindle at: {appdata_kindle}")
            print_warn("But TEMP.txt marker detected - this appears to be a leftover temp copy")
            print_step("Cleaning up leftover temp copy...")
            try:
                shutil.rmtree(appdata_kindle)
                print_ok("Cleanup completed")
                print()
                # Check Program Files again
                if os.path.exists(program_files_exe):
                    print_ok(f"Found Kindle at: {program_files_kindle}")
                    print_warn("KFXKeyExtractor requires Kindle in AppData location")
                    print()
                    print_step("Solution: Temporary copy will be created")
                    print("  - Source: " + program_files_kindle)
                    print("  - Destination: " + appdata_kindle)
                    print("  - This copy will be automatically deleted after extraction")
                    print("--------------------------------------------------")
                    print()
                    return program_files_kindle, True
                else:
                    print_error("No Kindle installation found after cleanup")
                    print("--------------------------------------------------")
                    print()
                    return None, False
            except Exception as e:
                print_error(f"Failed to cleanup temp copy: {e}")
                print("--------------------------------------------------")
                print()
                return None, False
    
    # CASE 3: Only Program Files location has Kindle.exe
    elif program_files_exists:
        print_ok(f"Found Kindle at: {program_files_kindle}")
        print_warn("Kindle is installed in Program Files (Global mode)")
        print_warn("KFXKeyExtractor requires Kindle in AppData location")
        print()
        print_step("Solution: Temporary copy will be created")
        print("  - Source: " + program_files_kindle)
        print("  - Destination: " + appdata_kindle)
        print("  - This copy will be automatically deleted after extraction")
        print("--------------------------------------------------")
        print()
        return program_files_kindle, True
    
    # CASE 4: Not found in either location
    else:
        print_error("Kindle.exe not found in expected locations:")
        print(f"  - {appdata_exe}")
        print(f"  - {program_files_exe}")
        print("--------------------------------------------------")
        print()
        return None, False

def launch_and_wait_for_kindle():
    """
    Launch Kindle.exe and allow user to press any key to continue
    User does not need to close Kindle - it's just a trigger if they want to use it
    Returns: (success: bool, error_message: str)
    """
    try:
        # Check if Kindle is already running
        already_running = is_kindle_running()
        
        if already_running:
            print_warn("Kindle is already running")
            print()
            print_ok("While Kindle for PC is Open Download The books you")
            print_ok("want to DeDRM and when ready Close Kindle for PC")
            print_ok("to begin the Key extraction")
            print()
            print_step("Waiting for Kindle to close...")
            print("  (Script will automatically continue when Kindle closes)")
            print()
            print_warn("Or press any key to skip waiting and continue immediately")
            print()
            
            # Wait for Kindle to close OR user to press a key
            while is_kindle_running():
                if msvcrt.kbhit():
                    msvcrt.getch()  # Consume the keypress
                    print_ok("User skipped waiting - continuing with script...")
                    print()
                    return True, ""
                time.sleep(1)  # Check every second
            
            print_ok("Kindle has closed - continuing with script...")
            print()
        else:
            # Find Kindle.exe location
            kindle_dir, _ = find_kindle_exe()
            
            if not kindle_dir:
                return False, "Kindle.exe not found. Please install Kindle for PC."
            
            kindle_exe = os.path.join(kindle_dir, "Kindle.exe")
            
            if not os.path.exists(kindle_exe):
                return False, f"Kindle.exe not found at: {kindle_exe}"
            
            print_step("Launching Kindle.exe...")
            print(f"  Location: {kindle_exe}")
            print()
            
            # Launch Kindle
            subprocess.Popen([kindle_exe])
            
            # Wait a moment for Kindle to start
            time.sleep(2)
            
            print_warn("Kindle launched successfully")
            print()
            print_ok("While Kindle for PC is Open Download The books you")
            print_ok("want to DeDRM and when ready Close Kindle for PC")
            print_ok("to begin the Key extraction")
            print()
            print_step("Waiting for Kindle to close...")
            print("  (Script will automatically continue when Kindle closes)")
            print()
            print_warn("Or press any key to skip waiting and continue immediately")
            print()
            
            # Wait for Kindle to close OR user to press a key
            while is_kindle_running():
                if msvcrt.kbhit():
                    msvcrt.getch()  # Consume the keypress
                    print_ok("User skipped waiting - continuing with script...")
                    print()
                    return True, ""
                time.sleep(1)  # Check every second
            
            print_ok("Kindle has closed - continuing with script...")
            print()
        
        return True, ""
        
    except Exception as e:
        return False, f"Failed to launch Kindle: {str(e)}"

def create_temp_kindle_copy(source_dir):
    """
    Create a temporary copy of Kindle installation in AppData
    Returns the path to the temporary copy
    """
    user_home = os.path.expanduser("~")
    dest_dir = os.path.join(user_home, "AppData", "Local", "Amazon", "Kindle", "application")
    temp_marker = os.path.join(dest_dir, "TEMP.txt")
    
    try:
        print_step("Creating temporary Kindle copy...")
        print(f"  Copying from: {source_dir}")
        print(f"  Copying to: {dest_dir}")
        print("  (This may take a minute...)")
        print()
        
        # Copy entire directory
        shutil.copytree(source_dir, dest_dir)
        print_ok("Kindle folder copied successfully")
        
        # Create marker file
        with open(temp_marker, 'w') as f:
            f.write(f"Temporary Kindle copy for key extraction\n")
            f.write(f"Created: {datetime.now()}\n")
            f.write(f"Source: {source_dir}\n")
        print_ok("Marker file created")
        print()
        
        return dest_dir
        
    except Exception as e:
        print_error(f"Failed to create temporary copy: {e}")
        # Cleanup partial copy if it exists
        if os.path.exists(dest_dir):
            try:
                shutil.rmtree(dest_dir)
            except:
                pass
        raise

def cleanup_temp_kindle_copy(kindle_dir):
    """
    Remove the temporary Kindle copy after extraction
    """
    temp_marker = os.path.join(kindle_dir, "TEMP.txt")
    
    # Only delete if it's a temp copy (has marker file)
    if os.path.exists(temp_marker):
        print_step("Cleaning up temporary Kindle copy...")
        try:
            shutil.rmtree(kindle_dir)
            print_ok("Temporary copy removed successfully")
        except Exception as e:
            print_error(f"Failed to cleanup temporary copy: {e}")
            print_warn(f"Please manually delete: {kindle_dir}")
        print()

def prevent_kindle_auto_update():
    """
    Create/update the 'updates' file to prevent Kindle for PC from auto-updating
    Only runs if Kindle is installed
    """
    user_home = os.path.expanduser("~")
    kindle_base = os.path.join(user_home, "AppData", "Local", "Amazon", "Kindle")
    updates_file = os.path.join(kindle_base, "updates")
    
    # Check if Kindle directory exists
    if not os.path.exists(kindle_base):
        return  # Kindle not installed, skip
    
    print_step("Configuring Kindle auto-update prevention...")
    
    try:
        # Create the updates file (no extension)
        with open(updates_file, 'w') as f:
            f.write("This file prevents Kindle for PC from auto-updating.\n")
            f.write("Created by Kindle Key Finder script.\n")
            f.write("Safe to delete if you want to allow auto-updates.\n")
        
        print_ok("Auto-update prevention file created/updated")
        print_ok(f"Location: {updates_file}")
        print("   Kindle for PC will not auto-update while this file exists")
    except Exception as e:
        print_warn(f"Could not create auto-update prevention file: {e}")
    
    print()

def fetch_book_title_from_asin(asin):
    """
    Fetch book title from ASIN using fetch-ebook-metadata command
    Returns: book title or ASIN if fetch fails
    """
    try:
        # Run fetch-ebook-metadata command
        result = subprocess.run(
            ['fetch-ebook-metadata', '-I', f'asin:{asin}'],
            capture_output=True,
            text=True,
            timeout=10,
            encoding='utf-8',
            errors='replace'
        )
        
        if result.returncode == 0 and result.stdout:
            # Parse output for title
            for line in result.stdout.split('\n'):
                if line.strip().startswith('Title'):
                    # Format: "Title               : The Sunken: A Dark Steampunk Fantasy"
                    parts = line.split(':', 1)
                    if len(parts) == 2:
                        title = parts[1].strip()
                        return title if title else asin
        
        # If parsing failed, return ASIN
        return asin
        
    except Exception:
        # If command fails, return ASIN
        return asin

def load_book_history(working_dir):
    """
    Load book processing history from history.txt
    Uses working_dir which respects fallback paths
    Returns: set of ASINs that have been previously processed
    """
    history_file = os.path.join(working_dir, "history.txt")
    processed_asins = set()
    
    try:
        if os.path.exists(history_file):
            with open(history_file, 'r', encoding='utf-8') as f:
                for line in f:
                    asin = line.strip()
                    if asin:
                        processed_asins.add(asin)
            print_ok(f"Loaded history: {len(processed_asins)} book(s) previously processed")
        else:
            print_step("No history file found - this appears to be first run")
    except Exception as e:
        print_warn(f"Failed to load history file: {e}")
    
    return processed_asins

def append_to_history(working_dir, asin):
    """
    Append ASIN to history.txt after successful extraction
    Uses working_dir which respects fallback paths
    """
    history_file = os.path.join(working_dir, "history.txt")
    
    try:
        # Ensure directory exists
        os.makedirs(os.path.dirname(history_file), exist_ok=True)
        
        with open(history_file, 'a', encoding='utf-8') as f:
            f.write(f"{asin}\n")
    except Exception as e:
        print_warn(f"Failed to update history file: {e}")

def prompt_history_action(total_books, previously_processed_count):
    """
    Prompt user what to do with previously processed books
    Returns: 'all' | 'new' | 'quit'
    """
    print()
    print_step("Book Processing History Check")
    print("--------------------------------------------------")
    print(f"Total books found: {total_books}")
    print(f"Previously processed: {previously_processed_count}")
    print(f"New books: {total_books - previously_processed_count}")
    print()
    print("Options:")
    print("  [A] Process All - Re-process everything including previously processed books")
    print("  [N] Process New Only - Skip previously processed books (recommended)")
    print("  [Q] Quit - Exit script")
    print()
    
    while True:
        choice = input("Your choice (A/N/Q) [N]: ").strip().upper()
        if choice == '':
            choice = 'N'  # Default to New Only
        if choice == 'A':
            print()
            print_ok("Will process all books including previously processed")
            print()
            return 'all'
        elif choice == 'N':
            print()
            print_ok("Will skip previously processed books")
            print()
            return 'new'
        elif choice == 'Q':
            print()
            print_warn("Script cancelled by user")
            return 'quit'
        else:
            print_error("Invalid choice. Please enter A, N, or Q.")

def scan_kindle_content_directory(content_dir):
    """
    Scan Kindle Content directory for individual book folders (recursively)
    Returns: list of tuples [(asin, book_folder_path, book_title), ...]
    """
    book_folders = []
    
    try:
        if not os.path.exists(content_dir):
            print_error(f"Content directory does not exist: {content_dir}")
            return []
        
        # Scan recursively for folders containing book files
        for root, dirs, files in os.walk(content_dir):
            # Check if this directory contains any book files
            book_files = [f for f in files if f.lower().endswith(('.azw', '.kfx', '.kfx-zip', '.azw3'))]
            
            if book_files:
                # This folder contains books - extract ASIN from folder name or first book file
                folder_name = os.path.basename(root)
                
                # Try to extract ASIN from folder name first (e.g., "B00N17VVZC_EBOK")
                if '_' in folder_name:
                    asin = folder_name.split('_')[0]
                else:
                    # Fallback: Extract from first book filename (e.g., "B00N17VVZC_EBOK.azw")
                    first_book = book_files[0]
                    asin = os.path.splitext(first_book)[0].split('_')[0]
                
                book_title = asin  # Default to ASIN
                book_folders.append((asin, root, book_title))
        
        return book_folders
        
    except Exception as e:
        print_error(f"Error scanning content directory: {e}")
        return []

def extract_keys_from_single_book(extractor_path, kindle_dir, book_folder, output_key, output_k4i, asin, book_title, working_dir=None, raw_log_path=None, book_prefix=""):
    """
    Extract keys from a single book folder using temporary directory workaround
    Returns: (success: bool, dsn: str, tokens: str, error_msg: str, asin: str)
    """
    # Initialize cmd at the very beginning to avoid unbound variable error in exception handlers
    cmd = []
    
    script_dir = os.path.dirname(os.path.abspath(__file__))
    user_home = os.path.expanduser("~")
    
    # Use working_dir if provided, otherwise determine it
    if working_dir is None:
        can_write, _ = check_write_permissions(script_dir)
        if can_write:
            working_dir = script_dir
        else:
            working_dir = os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    # Try to use working_dir temp_extraction, but fallback to Windows TEMP if access denied
    temp_dir = os.path.join(working_dir, "temp_extraction")
    
    # Test if we can actually create/access this directory
    try:
        os.makedirs(temp_dir, exist_ok=True)
        # Try to write a test file to verify access
        test_file = os.path.join(temp_dir, ".write_test")
        with open(test_file, 'w') as f:
            f.write("test")
        os.remove(test_file)
    except (OSError, PermissionError):
        # Fallback to Windows TEMP directory if AppData is inaccessible
        import tempfile
        temp_dir = os.path.join(tempfile.gettempdir(), "Kindle_Key_Finder_Extraction")
    
    try:
        
        # Copy extractor to Kindle folder if not already there
        extractor_in_kindle = os.path.join(kindle_dir, "KFXKeyExtractor28.exe")
        if not os.path.exists(extractor_in_kindle):
            shutil.copy2(extractor_path, kindle_dir)
        
        # Create temporary directory for this book only
        # Use robust cleanup that handles Windows file locking
        if os.path.exists(temp_dir):
            success = rmtree_with_retry(temp_dir, max_retries=5, retry_delay=0.5)
            if not success:
                return False, None, None, f"Failed to clean temp directory (Windows file lock): Try closing Kindle or other apps accessing these files", asin
        
        # Create temp directory
        try:
            os.makedirs(temp_dir, exist_ok=True)
        except Exception as e:
            return False, None, None, f"Failed to create temp directory: {e}", asin
        
        # Copy single book folder to temp directory
        book_name = os.path.basename(book_folder)
        temp_book = os.path.join(temp_dir, book_name)
        
        try:
            shutil.copytree(book_folder, temp_book)
        except (OSError, PermissionError) as e:
            # If copytree fails with access denied, provide detailed error
            error_details = f"Access denied copying book folder - {str(e)}"
            if "onedrive" in book_folder.lower():
                error_details += "\n          OneDrive sync may be blocking file operations"
                error_details += "\n          Try: Close OneDrive, disable real-time sync, or move books to C:\\Temp"
            return False, None, None, error_details, asin
        
        # Timer display thread (custom for extraction - reprints full line)
        # START TIMER FIRST - before subprocess creation
        timer_stopped = threading.Event()
        timeout_seconds = 60
        
        def display_extraction_timer():
            start_time = time.time()
            while not timer_stopped.is_set():
                elapsed = int(time.time() - start_time)
                minutes, seconds = divmod(elapsed, 60)
                timeout_mins, timeout_secs = divmod(timeout_seconds, 60)
                timer_str = f" [ {minutes:02d}:{seconds:02d} / {timeout_mins:02d}:{timeout_secs:02d} ]"
                
                # Reprint entire line: carriage return + book info + timer
                print(f"\r{book_prefix}{timer_str}", end='', flush=True)
                time.sleep(1)
        
        timer_thread = threading.Thread(target=display_extraction_timer, daemon=True)
        timer_thread.start()
        
        # NOW start subprocess - timer is already running
        # Run extractor on temp directory (not individual book folder)
        # This gives the extractor the directory structure it expects
        cmd = [extractor_in_kindle, temp_dir, output_key, output_k4i]
        process = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            bufsize=1,
            universal_newlines=True
        )
        
        stdout_lines = []
        stderr_lines = []
        
        # Read stderr in separate thread
        def read_stderr():
            try:
                stderr = process.stderr
                if stderr is None:
                    return
                for line in iter(stderr.readline, ''):
                    if line:
                        stderr_lines.append(line)
            except Exception:
                pass
        
        stderr_thread = threading.Thread(target=read_stderr, daemon=True)
        stderr_thread.start()
        
        # Read stdout
        try:
            stdout = process.stdout
            if stdout is not None:
                for line in iter(stdout.readline, ''):
                    if line:
                        stdout_lines.append(line)
        except Exception:
            pass
        
        # Wait for process to complete (with timeout)
        try:
            process.wait(timeout=timeout_seconds)
            # Stop timer, clear entire line, then reprint book line
            timer_stopped.set()
            timer_thread.join(timeout=1)
            # Clear line completely (100 chars to ensure timer is fully removed)
            print("\r" + " " * 100 + "\r", end='', flush=True)
            print(f"{book_prefix}", end='', flush=True)
        except subprocess.TimeoutExpired:
            # Stop timer, clear entire line, then reprint book line on timeout
            timer_stopped.set()
            timer_thread.join(timeout=1)
            # Clear line completely (100 chars to ensure timer is fully removed)
            print("\r" + " " * 100 + "\r", end='', flush=True)
            print(f"{book_prefix}", end='', flush=True)
            process.kill()
            # Log to raw log if enabled
            if raw_log_path:
                append_to_raw_log(
                    raw_log_path,
                    cmd,
                    ''.join(stdout_lines),
                    ''.join(stderr_lines),
                    -1,  # Timeout exit code
                    book_info=f"{asin} - {book_title}"
                )
            # Cleanup temp directory before returning (use robust cleanup)
            if os.path.exists(temp_dir):
                rmtree_with_retry(temp_dir)
            return False, None, None, f"Timeout after 60 seconds", asin
        
        # Wait for stderr thread
        stderr_thread.join(timeout=5)
        
        # Log to raw log if enabled
        if raw_log_path:
            append_to_raw_log(
                raw_log_path,
                cmd,
                ''.join(stdout_lines),
                ''.join(stderr_lines),
                process.returncode,
                book_info=f"{asin} - {book_title}"
            )
        
        # Check if extraction was successful
        if process.returncode != 0:
            # Combine stdout and stderr for complete error context
            stdout_text = ''.join(stdout_lines)
            stderr_text = ''.join(stderr_lines)
            
            # Filter out harmless Qt/Fontconfig messages but keep real errors
            error_lines = []
            for line in stderr_text.split('\n'):
                line_stripped = line.strip()
                if line_stripped and \
                   'QObject::startTimer' not in line and \
                   'Fontconfig error' not in line and \
                   'QThread' not in line:
                    error_lines.append(line_stripped)
            
            # If we have filtered errors, use them; otherwise generic message
            if error_lines:
                error_msg = '\n'.join(error_lines)
            else:
                # Check stdout for error messages
                if stdout_text and ('error' in stdout_text.lower() or 'failed' in stdout_text.lower()):
                    error_msg = stdout_text.strip()
                else:
                    error_msg = f"Key extraction failed (exit code {process.returncode})"
            
            # Cleanup temp directory before returning (use robust cleanup)
            if os.path.exists(temp_dir):
                rmtree_with_retry(temp_dir)
            return False, None, None, error_msg, asin
        
        # Parse output for DSN and tokens
        dsn = None
        tokens = None
        stdout_text = ''.join(stdout_lines)
        
        if stdout_text:
            lines = stdout_text.split('\n')
            for line in lines:
                if line.startswith('DSN '):
                    dsn = line.replace('DSN ', '').strip()
                elif line.startswith('Tokens '):
                    tokens_line = line.replace('Tokens ', '').strip()
                    if ',' in tokens_line:
                        tokens = tokens_line.split(',')[0].strip()
                    else:
                        tokens = tokens_line
        
        # Check if key files were generated
        if not os.path.exists(output_key) or not os.path.exists(output_k4i):
            # Cleanup temp directory before returning (use robust cleanup)
            if os.path.exists(temp_dir):
                rmtree_with_retry(temp_dir)
            return False, None, None, "Key files not generated", asin
        
        # Cleanup temp directory after successful extraction (use robust cleanup)
        if os.path.exists(temp_dir):
            rmtree_with_retry(temp_dir)
        
        return True, dsn, tokens, None, asin
        
    except Exception as e:
        # Cleanup temp directory on exception (use robust cleanup)
        if os.path.exists(temp_dir):
            rmtree_with_retry(temp_dir)
        return False, None, None, str(e), asin

def append_keys_to_files(output_key, output_k4i, temp_key, temp_k4i):
    """
    Append newly extracted keys to existing key files
    Avoids duplicates by checking if keys already exist
    Returns: (success: bool, error_msg: str)
    """
    try:
        # Check if temp files exist
        if not os.path.exists(temp_key) or not os.path.exists(temp_k4i):
            return False, "Temporary key files not found"
        
        # Read new keys from temp files
        with open(temp_key, 'r') as f:
            new_key_content = f.read().strip()
        
        with open(temp_k4i, 'r') as f:
            new_k4i_content = f.read().strip()
        
        # For kindlekey.txt: Append if not duplicate
        if os.path.exists(output_key):
            with open(output_key, 'r') as f:
                existing_key_content = f.read()
            
            # Check if this key already exists
            if new_key_content not in existing_key_content:
                with open(output_key, 'a') as f:
                    f.write('\n' + new_key_content)
        else:
            # Create new file
            with open(output_key, 'w') as f:
                f.write(new_key_content)
        
        # For kindlekey.k4i: Merge JSON data
        if os.path.exists(output_k4i):
            with open(output_k4i, 'r') as f:
                existing_k4i = json.load(f)
            
            new_k4i = json.loads(new_k4i_content)
            
            # Merge secrets arrays (avoid duplicates)
            for key in ['kindle.account.secrets', 'kindle.account.new_secrets', 'kindle.account.clear_old_secrets']:
                if key in new_k4i:
                    if key not in existing_k4i:
                        existing_k4i[key] = []
                    for item in new_k4i[key]:
                        if item not in existing_k4i[key]:
                            existing_k4i[key].append(item)
            
            # Update DSN if present
            if 'DSN' in new_k4i and new_k4i['DSN']:
                existing_k4i['DSN'] = new_k4i['DSN']
            
            # Update tokens if present
            if 'kindle.account.tokens' in new_k4i and new_k4i['kindle.account.tokens']:
                existing_k4i['kindle.account.tokens'] = new_k4i['kindle.account.tokens']
            
            # Write merged data
            with open(output_k4i, 'w') as f:
                json.dump(existing_k4i, f, indent=2)
        else:
            # Create new file
            with open(output_k4i, 'w') as f:
                f.write(new_k4i_content)
        
        return True, ""
        
    except Exception as e:
        return False, str(e)

def get_raw_log_path(working_dir, phase_name):
    """
    Get path for RAW debug log file
    Creates timestamped filename per script run
    
    Args:
        working_dir: Working directory (respects fallback paths)
        phase_name: Phase name (extraction, import, conversion)
    
    Returns:
        str: Full path to raw log file
    """
    logs_dir = os.path.join(working_dir, "Logs", f"{phase_name}_logs")
    os.makedirs(logs_dir, exist_ok=True)
    
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    return os.path.join(logs_dir, f"raw_{phase_name}_{timestamp}.log")

def cleanup_old_raw_logs(logs_dir, keep_count=10):
    """
    Cleanup old RAW log files, keeping only the most recent ones
    
    Args:
        logs_dir: Directory containing raw log files
        keep_count: Number of most recent files to keep (default: 10)
    """
    try:
        if not os.path.exists(logs_dir):
            return
        
        # Find all raw_*.log files
        raw_logs = []
        for filename in os.listdir(logs_dir):
            if filename.startswith('raw_') and filename.endswith('.log'):
                filepath = os.path.join(logs_dir, filename)
                mtime = os.path.getmtime(filepath)
                raw_logs.append((filepath, mtime))
        
        # Sort by modification time (newest first)
        raw_logs.sort(key=lambda x: x[1], reverse=True)
        
        # Delete files beyond keep_count
        for filepath, _ in raw_logs[keep_count:]:
            try:
                os.remove(filepath)
            except Exception:
                pass  # Ignore errors during cleanup
                
    except Exception:
        pass  # Silently ignore cleanup errors

def append_to_raw_log(raw_log_path, command, stdout, stderr, exit_code, book_info=None):
    """
    Append subprocess results to RAW debug log file
    Creates file with header if it doesn't exist
    Thread-safe atomic append operation
    
    Args:
        raw_log_path: Path to raw log file
        command: Command that was executed (list or string)
        stdout: Standard output from subprocess
        stderr: Standard error from subprocess
        exit_code: Process exit code
        book_info: Optional book identifier (e.g., ASIN or book name)
    """
    try:
        # Check if file exists to determine if we need header
        file_exists = os.path.exists(raw_log_path)
        
        with open(raw_log_path, 'a', encoding='utf-8') as f:
            # Write header if this is a new file
            if not file_exists:
                f.write("═" * 70 + "\n")
                phase_name = os.path.basename(os.path.dirname(raw_log_path)).replace('_logs', '').upper()
                f.write(f"RAW DEBUG LOG - {phase_name} PHASE\n")
                f.write("═" * 70 + "\n")
                f.write(f"Script Version: {SCRIPT_VERSION}\n")
                f.write(f"Log Created: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                f.write("═" * 70 + "\n\n")
            
            # Write entry separator
            f.write("─" * 70 + "\n")
            f.write(f"[{datetime.now().strftime('%H:%M:%S')}]")
            if book_info:
                f.write(f" Book: {book_info}\n")
            else:
                f.write("\n")
            
            # Write command
            if isinstance(command, list):
                f.write(f"Command: {' '.join(command)}\n")
            else:
                f.write(f"Command: {command}\n")
            
            f.write("─" * 70 + "\n")
            
            # Write stdout
            f.write("STDOUT:\n")
            if stdout:
                f.write(stdout)
                if not stdout.endswith('\n'):
                    f.write('\n')
            else:
                f.write("(empty)\n")
            f.write("\n")
            
            # Write stderr
            f.write("STDERR:\n")
            if stderr:
                f.write(stderr)
                if not stderr.endswith('\n'):
                    f.write('\n')
            else:
                f.write("(empty)\n")
            f.write("\n")
            
            # Write exit code
            f.write(f"Exit Code: {exit_code}\n")
            f.write("═" * 70 + "\n\n")
            
    except Exception:
        pass  # Silently ignore logging errors

def write_extraction_log(extraction_stats, working_dir):
    """
    Write detailed extraction log to file
    Uses working_dir which respects fallback paths
    Returns: log file path
    """
    # Create logs directory with extraction subfolder
    logs_dir = os.path.join(working_dir, "Logs", "extraction_logs")
    os.makedirs(logs_dir, exist_ok=True)
    
    # Create timestamped log file
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    log_file = os.path.join(logs_dir, f"extraction_{timestamp}.log")
    
    try:
        with open(log_file, 'w', encoding='utf-8') as f:
            # Header
            f.write("=" * 70 + "\n")
            f.write("KINDLE KEY EXTRACTION LOG\n")
            f.write("=" * 70 + "\n")
            f.write(f"Date: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
            f.write(f"Script Version: {SCRIPT_VERSION}\n")
            f.write(f"Total Books Found: {extraction_stats['total']}\n")
            f.write(f"Timeout per book: 60 seconds\n")
            f.write("\n")
            
            # Failed extractions section
            if extraction_stats.get('failed_books'):
                f.write("-" * 70 + "\n")
                f.write(f"FAILED EXTRACTIONS ({len(extraction_stats['failed_books'])})\n")
                f.write("-" * 70 + "\n")
                for asin, title, error_msg in extraction_stats['failed_books']:
                    f.write(f"\n[FAILED] {asin} - {title}\n")
                    f.write(f"  Error: {error_msg}\n")
                f.write("\n")
            
            # Success section (summary only)
            if extraction_stats['success'] > 0:
                f.write("-" * 70 + "\n")
                f.write(f"SUCCESSFUL EXTRACTIONS ({extraction_stats['success']})\n")
                f.write("-" * 70 + "\n")
                f.write(f"Keys successfully extracted from {extraction_stats['success']} book(s)\n")
                f.write("\n")
            
            # Summary
            f.write("=" * 70 + "\n")
            f.write("SUMMARY\n")
            f.write("=" * 70 + "\n")
            f.write(f"Total:    {extraction_stats['total']}\n")
            f.write(f"Success:  {extraction_stats['success']}\n")
            f.write(f"Failed:   {extraction_stats['failed']}\n")
            f.write("=" * 70 + "\n")
        
        return log_file
        
    except Exception as e:
        print_warn(f"Failed to write extraction log file: {e}")
        return None

def bulk_copy_selected_books(book_folders, staging_dir, source_content_dir):
    """
    Copy only the books that will be processed (post-history filtering) to staging directory
    
    Args:
        book_folders: List of (asin, book_path, title) tuples to process
        staging_dir: Target staging directory in AppData
        source_content_dir: Original content directory (in cloud)
    
    Returns:
        staging_dir path (becomes new content_dir)
    """
    # Clean old staging with robust retry logic (handles OneDrive file locks)
    if os.path.exists(staging_dir):
        print_step("Removing old staging...")
        success = rmtree_with_retry(staging_dir, max_retries=10, retry_delay=1.0)
        
        if success:
            print_ok("Old staging removed")
        else:
            print_error("Failed to remove old staging after multiple retries")
            print_warn("OneDrive may be syncing your AppData folder - this is the root cause")
            print_warn("SOLUTION: Close OneDrive completely, then re-run the script")
            print()
            raise Exception("Cannot proceed with dirty staging folder - OneDrive file lock detected")
        print()
    
    os.makedirs(staging_dir, exist_ok=True)
    
    print_step(f"Copying {len(book_folders)} book folder(s) to staging...")
    print()
    
    copy_success = 0
    copy_failed = 0
    
    for idx, (asin, book_path, title) in enumerate(book_folders, 1):
        folder_name = os.path.basename(book_path)
        staging_book_path = os.path.join(staging_dir, folder_name)
        
        print(f"[{idx}/{len(book_folders)}] {folder_name}...", end='', flush=True)
        
        try:
            shutil.copytree(book_path, staging_book_path)
            print(" ✓")
            copy_success += 1
        except Exception as e:
            print(f" ✗ Failed: {e}")
            copy_failed += 1
    
    print()
    print_ok(f"Copied {copy_success} book folder(s) successfully")
    
    if copy_failed > 0:
        print_warn(f"Failed to copy {copy_failed} book folder(s)")
    
    print()
    
    # Update book_folders to point to staging paths (modify list in-place)
    for i in range(len(book_folders)):
        asin, book_path, title = book_folders[i]
        folder_name = os.path.basename(book_path)
        book_folders[i] = (asin, os.path.join(staging_dir, folder_name), title)
    
    # Cooldown for cloud sync
    print_step("Waiting for cloud sync to stabilize (10 seconds)...")
    time.sleep(10)
    print_ok("Ready for processing!")
    print()
    
    return staging_dir

def extract_keys_using_extractor(extractor_path, content_dir, output_key, output_k4i, working_dir=None):
    """
    Extract keys using the KFXKeyExtractor28.exe with per-book processing
    Returns: (success: bool, dsn: str, tokens: str, extraction_stats: dict, updated_content_dir: str)
    """
    # Find kindle.exe location
    kindle_dir, needs_temp_copy = find_kindle_exe()
    
    if not kindle_dir:
        raise FileNotFoundError("Kindle for PC installation not found. Please install Kindle for PC.")
    
    temp_copy_created = False
    script_dir = os.path.dirname(os.path.abspath(__file__))
    user_home = os.path.expanduser("~")
    
    # Use provided working_dir or determine it
    if working_dir is None:
        can_write, _ = check_write_permissions(script_dir)
        if can_write:
            working_dir = script_dir
        else:
            working_dir = os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    # Initialize extraction statistics
    extraction_stats = {
        'total': 0,
        'success': 0,
        'failed': 0,
        'skipped': 0,
        'failed_books': [],  # List of tuples: (asin, title, error_msg)
        'skipped_books': []  # List of tuples: (asin, title, reason)
    }
    
    # Load config to check if book title fetching and raw logging are enabled
    config = load_config()
    fetch_titles = config.get('fetch_book_titles', False) if config else False
    enable_raw_logs = config.get('enable_raw_logs', False) if config else False
    
    # Initialize raw log path if enabled
    raw_log_path = None
    if enable_raw_logs:
        raw_log_path = get_raw_log_path(working_dir, 'extraction')
        # Cleanup old raw logs (keep 10 most recent)
        logs_dir = os.path.join(working_dir, "Logs", "extraction_logs")
        cleanup_old_raw_logs(logs_dir, keep_count=10)
    
    try:
        # Create temporary copy if needed
        if needs_temp_copy:
            kindle_dir = create_temp_kindle_copy(kindle_dir)
            temp_copy_created = True
        
        # Scan for individual book folders
        print_step("Scanning for book folders...")
        book_folders = scan_kindle_content_directory(content_dir)
        
        if not book_folders:
            print_warn("No book folders found in content directory")
            print()
            print("--------------------------------------------------")
            print()
            print_error("No books found in content directory!")
            print(f"  Selected path: {content_dir}")
            print()
            
            # Offer recovery workflow instead of exiting
            print_warn("The selected path does not contain any books.")
            print("This could happen if:")
            print("  - You haven't downloaded books to this location")
            print("  - You selected the wrong folder")
            print("  - Books are in a different location")
            print()
            
            print("Options:")
            print("  [R] Retry - Choose a different Kindle content path")
            print("  [Q] Quit - Exit script")
            print()
            
            while True:
                choice = input("Your choice (R/Q): ").strip().upper()
                if choice == 'R':
                    print()
                    print_step("Re-selecting Kindle content path...")
                    print()
                    
                    # Get new path from user
                    user_home = os.path.expanduser("~")
                    default_content = os.path.join(user_home, "Documents", "My Kindle Content")
                    new_content_dir = get_kindle_content_path(default_content, use_timer=False)
                    
                    # Update config file with new path
                    current_config = load_config()
                    if current_config:
                        current_config['kindle_content_path'] = new_content_dir
                        save_config(current_config)
                        print_ok("Configuration updated with new path")
                        print()
                    
                    # Update content_dir for retry
                    content_dir = new_content_dir
                    
                    # Retry scanning with new path
                    print_step("Rescanning for book folders...")
                    book_folders = scan_kindle_content_directory(content_dir)
                    
                    if not book_folders:
                        print_warn("Still no book folders found in new directory")
                        print()
                        print("Would you like to try a different path?")
                        continue  # Loop back to offer choices again
                    else:
                        # Success! Books found with new path
                        print_ok(f"Found {len(book_folders)} book folder(s) with new path")
                        print()
                        break  # Exit the retry loop and continue with extraction
                        
                elif choice == 'Q':
                    print()
                    print_warn("Script cancelled by user")
                    print()
                    # Show final credits page before exit
                    display_credits_and_links()
                    sys.exit(0)
                else:
                    print_error("Invalid choice. Please enter R or Q.")
        
        extraction_stats['total'] = len(book_folders)
        print_ok(f"Found {len(book_folders)} book folder(s)")
        print()
        
        # Load book history
        processed_asins = load_book_history(working_dir)
        
        # Check if any books were previously processed
        previously_processed = [asin for asin, _, _ in book_folders if asin in processed_asins]
        
        if previously_processed:
            # Prompt user for action
            action = prompt_history_action(len(book_folders), len(previously_processed))
            
            if action == 'quit':
                # User chose to quit - exit gracefully without errors
                print()
                return 0
            elif action == 'new':
                # Track which books are being skipped due to history
                skipped = [(asin, title, "Previously processed") 
                          for asin, _, title in book_folders 
                          if asin in processed_asins]
                extraction_stats['skipped_books'] = skipped
                extraction_stats['skipped'] = len(skipped)
                
                # Filter out previously processed books
                book_folders = [(asin, path, title) for asin, path, title in book_folders if asin not in processed_asins]
                print_ok(f"Processing {len(book_folders)} new book(s)")
                print()
            # else action == 'all', process everything
        
        if not book_folders:
            print_ok("No new books to process - all books have been previously processed")
            print()
            print_step("Script will now exit gracefully")
            print()
            # Return success (not failure) since this is an expected scenario
            return True, None, None, extraction_stats
        
        # NEW: Check if content_dir (books location) is in a cloud-synced location
        is_cloud, cloud_service = is_cloud_synced_location(content_dir)
        
        if is_cloud:
            print()
            print_warn(f"Books detected in cloud location: {cloud_service}")
            print(f"  Book location: {content_dir}")
            print()
            print_colored("═" * 70, 'yellow')
            print_warn("CLOUD SYNC DETECTION - STAGING REQUIRED")
            print_colored("═" * 70, 'yellow')
            print()
            print("Cloud folders can cause 'Access Denied' errors during book processing.")
            print("To ensure reliable extraction, books will be staged to local directory.")
            print()
            print("This will:")
            print("  1. Copy books to AppData staging folder")
            print("  2. Wait 10 seconds for cloud sync to stabilize")
            print("  3. Process books from local staging (no cloud conflicts)")
            print("  4. Clean up staging folder after extraction")
            print()
            
            # Create staging directory in working_dir
            staging_dir = os.path.join(working_dir, "temp_kindle_content")
            
            # Bulk copy only the books that will be processed (already filtered by history)
            content_dir = bulk_copy_selected_books(book_folders, staging_dir, content_dir)
            
            print_ok("Books staged successfully!")
            print_ok(f"Processing from: {staging_dir}")
            print()
            
            # IMPORTANT: Display warning about potential WinError 5 (Access Denied)
            print()
            print_colored("╔" + "═" * 68 + "╗", 'red')
            print_colored("║" + " " * 68 + "║", 'red')
            print_colored("║" + "⚠  WARNING: CLOUD SYNC ACCESS ISSUES - READ CAREFULLY".center(68) + "║", 'red')
            print_colored("║" + " " * 68 + "║", 'red')
            print_colored("╚" + "═" * 68 + "╝", 'red')
            print()
            print("The script will now ATTEMPT to work around cloud sync access issues.")
            print_warn("However, this workaround is HIT or MISS and may still fail!")
            print()
            print_colored("IF THE SCRIPT FAILS WITH 'ACCESS DENIED' OR 'WinError 5':", 'yellow')
            print()
            print("TRY:")
            print("   1. Pausing or Closing your Cloud Service application")
            print('   2. try running script again')
            print()
            print("RECOMMENDED SOLUTION:")
            print("  1. Open Kindle for PC")
            print("  2. Go to Tools → Options → Content")
            print("  3. Change the download location to a non-cloud folder")
            print(f"     Example: C:\\Temp\\Kindle Books (NOT {cloud_service})")
            print("  4. Re-download your books to the new location")
            print("  5. Run this script again pointing to the new location")
            print()
            print("ALTERNATIVE (TEMPORARY FIX):")
            print("  1. Copy/Move the book folders that need DeDRM to a local folder")
            print(f"     Example: Move from {content_dir}")
            print("              to C:\\Temp\\Kindle Books")
            print("  2. Run this script pointing to the new local location")
            print()
            print_warn("Cloud sync (OneDrive, Google Drive, etc.) actively blocks file access!")
            print("This is especially common with OneDrive's 'Files On-Demand' feature.")
            print()
        
        # Process each book individually
        print_step(f"Extracting keys from {len(book_folders)} book(s)...")
        print()
        
        # Create temporary key files for per-book extraction
        temp_key = output_key + ".temp"
        temp_k4i = output_k4i + ".temp"
        
        dsn = None
        tokens = None
        
        for idx, (asin, book_folder, book_title) in enumerate(book_folders, 1):
            # Get folder name (e.g., "B00JBVUJM8_EBOK")
            folder_name = os.path.basename(book_folder)
            
            # Build book prefix for timer display based on fetch_titles configuration
            if fetch_titles:
                fetched_title = fetch_book_title_from_asin(asin)
                book_prefix = f"[{idx}/{len(book_folders)}] {folder_name} - {fetched_title}..."
            else:
                book_prefix = f"[{idx}/{len(book_folders)}] {folder_name}..."
            
            # Extract keys from single book (pass book_prefix for timer display)
            success, book_dsn, book_tokens, error_msg, _ = extract_keys_from_single_book(
                extractor_path, kindle_dir, book_folder, temp_key, temp_k4i, asin, book_title, 
                working_dir=working_dir, raw_log_path=raw_log_path, book_prefix=book_prefix
            )
            
            if success:
                print_colored(" [OK] ✓", 'green')
                extraction_stats['success'] += 1
                
                # Store DSN and tokens from first successful extraction
                if not dsn and book_dsn:
                    dsn = book_dsn
                if not tokens and book_tokens:
                    tokens = book_tokens
                
                # Append keys to main files
                append_success, append_error = append_keys_to_files(output_key, output_k4i, temp_key, temp_k4i)
                if not append_success:
                    print_warn(f" (Warning: Failed to append keys - {append_error})")
                
                # Update history with successfully processed book
                append_to_history(working_dir, asin)
                
                # Cleanup temp files
                try:
                    if os.path.exists(temp_key):
                        os.remove(temp_key)
                    if os.path.exists(temp_k4i):
                        os.remove(temp_k4i)
                except Exception:
                    pass
            else:
                print_colored(" [ERROR] ✗ FAILED", 'red')
                extraction_stats['failed'] += 1
                
                # Fetch book title from ASIN for better error reporting
                fetched_title = fetch_book_title_from_asin(asin)
                extraction_stats['failed_books'].append((asin, fetched_title, error_msg if error_msg else "Unknown error"))
        
        print()
        
        # Cleanup extractor from Kindle folder
        extractor_in_kindle = os.path.join(kindle_dir, "KFXKeyExtractor28.exe")
        if os.path.exists(extractor_in_kindle):
            os.remove(extractor_in_kindle)
            print_ok("Extractor cleaned up from Kindle folder")
        
        # Write extraction log if there were failures
        if extraction_stats['failed'] > 0:
            log_file = write_extraction_log(extraction_stats, working_dir)
            if log_file:
                print_step(f"Extraction error log saved to:")
                print(f"      {log_file}")
        
        # Show raw log location if enabled
        if enable_raw_logs and raw_log_path:
            print_step(f"Raw debug log saved to:")
            print(f"      {raw_log_path}")
        
        print()
        
        # Check if extraction was successful (at least one book succeeded)
        overall_success = extraction_stats['success'] > 0
        
        # Return updated content_dir so caller can use the correct path
        return overall_success, dsn, tokens, extraction_stats, content_dir
        
    except Exception as e:
        print_error(f"Extractor method failed: {e}")
        # Return original content_dir even on error
        return False, None, None, extraction_stats, content_dir
        
    finally:
        # Always cleanup temporary copy if we created one
        if temp_copy_created:
            cleanup_temp_kindle_copy(kindle_dir)

def create_kindle_key_from_k4i(k4i_path, dsn=None, tokens=None):
    """
    Create a Kindle key entry exactly like the DeDRM plugin does
    This replicates the logic from kindlekey.py's kindlekeys() function
    Handles missing fields gracefully with extracted or default values
    """
    try:
        with open(k4i_path, 'r') as f:
            k4i_data = json.load(f)
        
        # Handle missing fields by providing defaults or extracted values
        kindle_key = {
            "DSN": k4i_data.get("DSN", dsn or ""),
            "kindle.account.clear_old_secrets": k4i_data.get("kindle.account.clear_old_secrets", []),
            "kindle.account.new_secrets": k4i_data.get("kindle.account.new_secrets", []), 
            "kindle.account.secrets": k4i_data.get("kindle.account.secrets", []),
            "kindle.account.tokens": k4i_data.get("kindle.account.tokens", tokens or "")
        }
        
        return kindle_key
        
    except Exception as e:
        print_error(f"Failed to process k4i file: {e}")
        return None

def create_dedrm_config(kindle_key, kindlekey_txt_path, reference_json_path=None):
    """
    Create the dedrm.json configuration exactly like the plugin does
    """
    
    if reference_json_path and os.path.exists(reference_json_path):
        # Use reference file as template
        print_step("Using reference file as template...")
        with open(reference_json_path, 'r') as f:
            dedrm_config = json.load(f)
    else:
        # Create new structure matching plugin's default
        print_step("Creating new configuration structure...")
        dedrm_config = {
            "adeptkeys": {},
            "adobe_pdf_passphrases": [],
            "adobewineprefix": "",
            "androidkeys": {},
            "bandnkeys": {},
            "configured": True,
            "deobfuscate_fonts": True,
            "ereaderkeys": {},
            "kindleextrakeyfile": "",
            "kindlekeys": {},
            "kindlewineprefix": "",
            "lcp_passphrases": [],
            "pids": [],
            "remove_watermarks": False,
            "serials": []
        }
    
    # Set the extra key file path (with proper escaping for JSON)
    dedrm_config["kindleextrakeyfile"] = kindlekey_txt_path
    
    # Add the kindle key exactly like the plugin does
    # The plugin uses the key name "kindlekey" + count, but for single key we use "kindlekey"
    dedrm_config["kindlekeys"]["kindlekey"] = kindle_key
    
    return dedrm_config

# ============================================================================
# UNIFIED CONFIGURATION FUNCTIONS
# ============================================================================

# ============================================================================
# WRITE PERMISSIONS & PATH DISCOVERY FUNCTIONS
# ============================================================================

def is_cloud_synced_location(directory):
    """
    Detect if directory is inside a cloud sync folder
    Returns: (is_cloud: bool, cloud_service: str or None)
    """
    dir_lower = directory.lower()
    
    # Common cloud sync folder patterns
    cloud_patterns = {
        'onedrive': ['onedrive', '\\onedrive\\', '/onedrive/'],
        'google-drive': ['google drive', '\\google drive\\', '/google drive/', 'googledrive'],
        'dropbox': ['dropbox', '\\dropbox\\', '/dropbox/'],
        'icloud': ['icloud', '\\icloud\\', '/icloud/', 'icloud drive'],
        'box': ['\\box sync\\', '/box sync/', '\\box\\', '/box/'],
        'sync': ['sync.com', '\\sync\\', '/sync/']  # Sync.com
    }
    
    for service, patterns in cloud_patterns.items():
        for pattern in patterns:
            if pattern in dir_lower:
                return True, service
    
    return False, None

def check_write_permissions(directory):
    """
    Enhanced write permission check:
    1. Detects cloud sync folders proactively (OneDrive, Google Drive, etc.)
    2. Tests actual directory creation (not just files)
    
    Cloud folders can cause intermittent "Access Denied" errors during
    temp directory creation due to sync conflicts.
    
    Returns: (can_write: bool, error_msg: str or None)
    """
    # First check: Is this a cloud-synced location?
    is_cloud, cloud_service = is_cloud_synced_location(directory)
    if is_cloud:
        return False, f"Cloud sync folder detected ({cloud_service}) - may cause access issues"
    
    # Second check: Can we actually create a directory here?
    test_dir = os.path.join(directory, ".write_test_temp_dir")
    try:
        os.makedirs(test_dir, exist_ok=True)
        os.rmdir(test_dir)
        return True, None
    except Exception as e:
        return False, str(e)

def get_disk_space(path):
    """
    Get disk space information for a given path
    Returns: (free_space_gb: float, total_space_gb: float, percent_free: float) or (None, None, None) on error
    """
    try:
        # Ensure path exists or use parent directory
        check_path = path
        while not os.path.exists(check_path) and check_path != os.path.dirname(check_path):
            check_path = os.path.dirname(check_path)
        
        if not os.path.exists(check_path):
            return None, None, None
        
        stat = shutil.disk_usage(check_path)
        free_gb = stat.free / (1024 ** 3)  # Convert bytes to GB
        total_gb = stat.total / (1024 ** 3)
        percent_free = (stat.free / stat.total) * 100 if stat.total > 0 else 0
        
        return free_gb, total_gb, percent_free
    except Exception as e:
        print_warn(f"Could not check disk space for {path}: {e}")
        return None, None, None

def get_fallback_paths(user_home):
    """
    Generate fallback paths in %APPDATA% when script dir is not writable
    Returns: dict with all necessary paths
    """
    appdata_base = os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    return {
        'base_dir': appdata_base,
        'keys_dir': os.path.join(appdata_base, "Keys"),
        'logs_dir': os.path.join(appdata_base, "Logs"),
        'backups_dir': os.path.join(appdata_base, "backups"),
        'temp_extraction_dir': os.path.join(appdata_base, "temp_extraction"),
        'config_file': os.path.join(appdata_base, "key_finder_config.json"),
        'history_file': os.path.join(appdata_base, "history.txt")
    }

def discover_config_location(script_dir, user_home):
    """
    Discover which location has the config file and which is writable
    Returns: (config_path, working_dir, needs_migration, is_fallback)
    
    Priority logic:
    1. If script_dir is writable AND has config → use script_dir
    2. If script_dir is writable but no config, check fallback → migrate if found
    3. If script_dir NOT writable, use fallback (migrate if needed)
    """
    fallback_base = os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    fallback_config = os.path.join(fallback_base, "key_finder_config.json")
    script_config = os.path.join(script_dir, "key_finder_config.json")
    
    # Check if script_dir is writable
    can_write, error = check_write_permissions(script_dir)
    
    if can_write:
        # Script dir is writable - prefer it
        
        # Priority 1: Config exists in script_dir → use it
        if os.path.exists(script_config):
            return script_config, script_dir, False, False
        
        # Priority 2: Config exists in fallback → migrate to script_dir
        if os.path.exists(fallback_config):
            return fallback_config, script_dir, True, False
        
        # Priority 3: No config found → create in script_dir
        return script_config, script_dir, False, False
    else:
        # Script dir NOT writable - must use fallback
        
        # Priority 1: Config exists in fallback → use it
        if os.path.exists(fallback_config):
            return fallback_config, fallback_base, False, True
        
        # Priority 2: Config exists in script_dir → migrate to fallback
        if os.path.exists(script_config):
            return script_config, fallback_base, True, True
        
        # Priority 3: No config found → create in fallback
        return fallback_config, fallback_base, False, True

def create_location_marker(working_dir, script_dir):
    """
    Create _LOCATION_INFO.txt file to help users find their files
    """
    marker_path = os.path.join(working_dir, "_LOCATION_INFO.txt")
    
    try:
        with open(marker_path, 'w', encoding='utf-8') as f:
            f.write("=" * 70 + "\n")
            f.write("KINDLE KEY FINDER - FILE LOCATION NOTICE\n")
            f.write("=" * 70 + "\n\n")
            
            f.write(f"Created: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n\n")
            
            f.write("WHY ARE FILES HERE?\n")
            f.write("-" * 70 + "\n")
            f.write("The script detected that your original script location is not writable.\n")
            f.write("This commonly occurs when running from:\n")
            f.write("  • Network drives\n")
            f.write("  • Read-only folders\n")
            f.write("  • Restricted permissions locations\n\n")
            
            f.write(f"Original script location: {script_dir}\n")
            f.write(f"Current working location: {working_dir}\n\n")
            
            f.write("WHAT FILES ARE STORED HERE?\n")
            f.write("-" * 70 + "\n")
            f.write("  • Configuration file (key_finder_config.json)\n")
            f.write("  • Extracted keys (Keys/kindlekey.txt, Keys/kindlekey.k4i)\n")
            f.write("  • Processing logs (Logs/extraction_logs/, conversion_logs/, etc.)\n")
            f.write("  • Book history (history.txt)\n")
            f.write("  • Backups (backups/dedrm_backup_*.json)\n\n")
            
            f.write("THIS IS NORMAL AND SAFE\n")
            f.write("-" * 70 + "\n")
            f.write("Your files are safely stored in your Windows AppData Local folder.\n")
            f.write("This is a standard location for application data and is backed up\n")
            f.write("with your user profile.\n\n")
            
            f.write("TO ACCESS FILES:\n")
            f.write("-" * 70 + "\n")
            f.write("1. Press Win+R to open Run dialog\n")
            f.write("2. Type: %LOCALAPPDATA%\\Kindle_Key_Finder\n")
            f.write("3. Press Enter\n\n")
            
            f.write("=" * 70 + "\n")
        
        print_ok(f"Location info file created: _LOCATION_INFO.txt")
        
    except Exception as e:
        print_warn(f"Could not create location marker: {e}")

def migrate_config_to_fallback(source_config, dest_config, script_dir, fallback_base):
    """
    Migrate config and related files from script_dir to fallback location
    Also creates _LOCATION_INFO.txt marker
    """
    try:
        # Ensure fallback directory exists
        os.makedirs(os.path.dirname(dest_config), exist_ok=True)
        
        print_step("Migrating configuration to AppData...")
        
        # Copy config file
        if os.path.exists(source_config):
            shutil.copy2(source_config, dest_config)
            print_ok(f"Config migrated: {os.path.basename(dest_config)}")
        
        # Migrate other files if they exist
        files_to_migrate = [
            ('history.txt', 'History file'),
            ('Keys/kindlekey.txt', 'Kindle key'),
            ('Keys/kindlekey.k4i', 'Kindle account data')
        ]
        
        for rel_path, description in files_to_migrate:
            source_file = os.path.join(script_dir, rel_path)
            dest_file = os.path.join(fallback_base, rel_path)
            
            if os.path.exists(source_file):
                os.makedirs(os.path.dirname(dest_file), exist_ok=True)
                shutil.copy2(source_file, dest_file)
                print_ok(f"{description} migrated")
        
        # Create location marker file
        create_location_marker(fallback_base, script_dir)
        
        print()
        print_ok("Migration completed successfully")
        print()
        
        return True
        
    except Exception as e:
        print_error(f"Migration failed: {e}")
        print_warn("Will attempt to use fallback location without migration")
        return False

# ============================================================================
# COMPONENT VALIDATION FUNCTIONS
# ============================================================================

def get_calibre_plugins_dir():
    """Get Calibre plugins directory path"""
    user_home = os.path.expanduser("~")
    return os.path.join(user_home, "AppData", "Roaming", "calibre", "plugins")

def check_plugin_installed(plugin_name):
    """
    Check if a Calibre plugin is installed
    Returns: True if plugin exists, False otherwise
    """
    plugins_dir = get_calibre_plugins_dir()
    plugin_path = os.path.join(plugins_dir, f"{plugin_name}.zip")
    return os.path.exists(plugin_path)

def check_calibre_installed():
    """
    Check if Calibre is installed by testing calibredb command
    Returns: True if Calibre is installed, False otherwise
    """
    try:
        result = subprocess.run(
            ['calibredb', '--version'],
            capture_output=True,
            timeout=5
        )
        return result.returncode == 0
    except:
        return False

def check_extractor_exists(script_dir):
    """
    Check if KFXKeyExtractor28.exe exists in script directory
    Returns: (exists: bool, extractor_path: str)
    """
    extractor_path = os.path.join(script_dir, "KFXKeyExtractor28.exe")
    return (os.path.exists(extractor_path), extractor_path)

def validate_kindle_installation():
    """
    Check if Kindle for PC is installed
    Returns: (is_installed: bool, kindle_path: str or None)
    """
    kindle_dir, _ = find_kindle_exe()
    return (kindle_dir is not None, kindle_dir)

def validate_all_requirements(script_dir, user_home):
    """
    Comprehensive validation of all requirements
    Returns: validation_report dict
    """
    report = {
        'write_permissions': False,
        'working_dir': None,
        'is_fallback': False,
        'fallback_reason': None,
        'extractor_installed': False,
        'extractor_path': None,
        'kindle_installed': False,
        'kindle_path': None,
        'calibre_installed': False,
        'kfx_plugin_installed': False,
        'dedrm_plugin_installed': False,
        'all_critical_met': False,
        'warnings': []
    }
    
    # 1. Check write permissions & discover config location (critical)
    config_path, working_dir, needs_migration, is_fallback = discover_config_location(script_dir, user_home)
    report['write_permissions'] = True  # If we get here, we have SOMEWHERE to write
    report['working_dir'] = working_dir
    report['is_fallback'] = is_fallback
    
    if is_fallback:
        can_write, error = check_write_permissions(script_dir)
        report['fallback_reason'] = f"Script directory not writable: {error}"
    
    # 2. Check for KFXKeyExtractor28.exe (CRITICAL - needed for Phase 1)
    extractor_ok, extractor_path = check_extractor_exists(script_dir)
    report['extractor_installed'] = extractor_ok
    report['extractor_path'] = extractor_path
    if not extractor_ok:
        report['warnings'].append({
            'component': 'KFXKeyExtractor28.exe',
            'severity': 'CRITICAL',
            'impact': 'Cannot extract keys (Phase 1 will fail immediately)',
            'error_example': 'File not found: KFXKeyExtractor28.exe',
            'install_url': 'https://github.com/Satsuoni/DeDRM_tools'
        })
    
    # 3. Check Kindle (critical for Phase 1)
    kindle_ok, kindle_path = validate_kindle_installation()
    report['kindle_installed'] = kindle_ok
    report['kindle_path'] = kindle_path
    if not kindle_ok:
        report['warnings'].append({
            'component': 'Kindle for PC',
            'severity': 'CRITICAL',
            'impact': 'Cannot extract keys (Phase 1 will fail)',
            'install_url': 'https://www.amazon.com/kindle-dbs/fd/kcp'
        })
    
    # 4. Check Calibre (critical for Phases 3-4)
    calibre_ok = check_calibre_installed()
    report['calibre_installed'] = calibre_ok
    if not calibre_ok:
        report['warnings'].append({
            'component': 'Calibre',
            'severity': 'WARNING',
            'impact': 'Cannot import/convert books (Phases 3-4 will be skipped)',
            'install_url': 'https://calibre-ebook.com/download'
        })
    
    # 5 & 6. Check plugins (only if Calibre is installed)
    if calibre_ok:
        kfx_ok = check_plugin_installed('KFX Input')
        dedrm_ok = check_plugin_installed('DeDRM')
        
        report['kfx_plugin_installed'] = kfx_ok
        report['dedrm_plugin_installed'] = dedrm_ok
        
        if not kfx_ok:
            report['warnings'].append({
                'component': 'KFX Input Plugin',
                'severity': 'WARNING',
                'impact': 'KFX books cannot be converted (will fail during conversion)',
                'error_example': 'This is an Amazon KFX book. It cannot be processed.',
                'install_url': 'https://www.mobileread.com/forums/showthread.php?t=283371'
            })
        
        if not dedrm_ok:
            report['warnings'].append({
                'component': 'DeDRM Plugin',
                'severity': 'WARNING',
                'impact': 'Imported books will remain DRM-protected',
                'error_example': 'Book has DRM and cannot be converted',
                'install_url': 'https://github.com/Satsuoni/DeDRM_tools'
            })
    
    # Determine if critical requirements are met (both extractor AND Kindle required)
    report['all_critical_met'] = extractor_ok and kindle_ok
    
    return report

def display_validation_results(report):
    """
    Display validation results with color-coded warnings
    Returns: True if user wants to continue, False to exit
    """
    print_step("Pre-Flight System Validation")
    print("=" * 70)
    print()
    
    # Validation Results Table
    print("┌─────────────────────────────┬────────────┬────────────────────────────────────────┐")
    print("│ Component                   │ Status     │ Details                                │")
    print("├─────────────────────────────┼────────────┼────────────────────────────────────────┤")
    
    # Row 1: Write Permissions
    if report['is_fallback']:
        status = "⚠ Fallback"
        details = "Using AppData location"
    else:
        status = "✓ OK"
        details = "Script directory writable"
    print(f"│ Write Permissions           │ {status:<10} │ {details:<38} │")
    
    # Row 2: KFXKeyExtractor28.exe
    if report['extractor_installed']:
        status = "✓ Found"
        details = "Required for Phase 1"
    else:
        status = "✗ Missing"
        details = "CRITICAL - Phase 1 will fail"
    print(f"│ KFXKeyExtractor28.exe       │ {status:<10} │ {details:<38} │")
    
    # Row 3: Kindle for PC
    if report['kindle_installed']:
        status = "✓ Found"
        # Truncate path if too long
        path_display = os.path.basename(report['kindle_path']) if report['kindle_path'] else "Installed"
        if len(path_display) > 38:
            path_display = path_display[:35] + "..."
        details = path_display
    else:
        status = "✗ Missing"
        details = "CRITICAL - Phase 1 will fail"
    print(f"│ Kindle for PC               │ {status:<10} │ {details:<38} │")
    
    # Row 4: Calibre
    if report['calibre_installed']:
        status = "✓ Found"
        details = "Required for Phases 3-4"
    else:
        status = "✗ Missing"
        details = "Phases 3-4 will be skipped"
    print(f"│ Calibre                     │ {status:<10} │ {details:<38} │")
    
    # Row 5: KFX Input Plugin
    if report['kfx_plugin_installed']:
        status = "✓ Found"
        details = "Converts KFX books"
    else:
        status = "✗ Missing"
        details = "KFX conversion will fail"
    print(f"│ KFX Input Plugin            │ {status:<10} │ {details:<38} │")
    
    # Row 6: DeDRM Plugin
    if report['dedrm_plugin_installed']:
        status = "✓ Found"
        details = "Removes DRM protection"
    else:
        status = "✗ Missing"
        details = "Books will remain DRM-protected"
    print(f"│ DeDRM Plugin                │ {status:<10} │ {details:<38} │")
    
    # Table footer
    print("└─────────────────────────────┴────────────┴────────────────────────────────────────┘")
    
    # Storage Information Section
    print()
    print("┌─────────────────────────────┬────────────┬────────────────────────────────────────┐")
    print("│ Storage Information         │ Status     │ Details                                │")
    print("├─────────────────────────────┼────────────┼────────────────────────────────────────┤")
    
    # Get user home for AppData check
    user_home = os.path.expanduser("~")
    appdata_local = os.path.join(user_home, "AppData", "Local")
    
    # Check AppData Local storage
    appdata_free, appdata_total, appdata_percent = get_disk_space(appdata_local)
    if appdata_free is not None:
        storage_display = f"{appdata_free:.1f} GB / {appdata_total:.1f} GB ({appdata_percent:.1f}% free)"
        # Color code based on available space
        if appdata_free < 1.0:
            storage_status = "⚠ CRITICAL"
        elif appdata_free < 5.0:
            storage_status = "⚠ Warning"
        else:
            storage_status = "✓ OK"
        print(f"│ AppData Free Space          │ {storage_status:<10} │ {storage_display:<38} │")
    else:
        print(f"│ AppData Free Space          │ {'Unknown':<10} │ {'N/A':<38} │")
    
    # Check Calibre Library storage (if configured)
    # Load config to check if Calibre import is enabled
    current_config = load_config()
    if current_config and 'calibre_import' in current_config and current_config['calibre_import'].get('enabled'):
        lib_path = current_config['calibre_import'].get('library_path')
        if lib_path and lib_path != 'Not set':
            lib_free, lib_total, lib_percent = get_disk_space(lib_path)
            if lib_free is not None:
                storage_display = f"{lib_free:.1f} GB / {lib_total:.1f} GB ({lib_percent:.1f}% free)"
                # Color code based on available space
                if lib_free < 1.0:
                    storage_status = "⚠ CRITICAL"
                elif lib_free < 5.0:
                    storage_status = "⚠ Warning"
                else:
                    storage_status = "✓ OK"
                print(f"│ Calibre Library Free Space  │ {storage_status:<10} │ {storage_display:<38} │")
            else:
                print(f"│ Calibre Library Free Space  │ {'Unknown':<10} │ {'N/A':<38} │")
    
    print("└─────────────────────────────┴────────────┴────────────────────────────────────────┘")
    print()
    
    # Show fallback location details if applicable  
    if report['is_fallback']:
        print_warn("IMPORTANT: Using fallback location for file operations")
        print(f"   Reason: {report['fallback_reason']}")
        print(f"   Working directory: {report['working_dir']}")
        print()
        # Check if it's specifically a cloud folder issue
        if 'cloud' in report['fallback_reason'].lower():
            print_colored("   ⚠  CLOUD SYNC FOLDER DETECTED", 'yellow')
            print("   Cloud folders (OneDrive, Google Drive, etc.) can cause:")
            print("   - Intermittent 'Access Denied' errors during temp file creation")
            print("   - File sync conflicts during extraction")
            print("   - Unreliable directory creation")
            print()
            print("   All files will be safely stored in:")
            print(f"   {report['working_dir']}")
            print()
    
    # Display warnings if any
    if report['warnings']:
        print("=" * 70)
        print_warn(f"VALIDATION WARNINGS ({len(report['warnings'])})")
        print("=" * 70)
        print()
        
        for warning in report['warnings']:
            severity_color = 'red' if warning['severity'] == 'CRITICAL' else 'yellow'
            print_colored(f"[{warning['severity']}] {warning['component']}", severity_color)
            print(f"   Impact: {warning['impact']}")
            if 'error_example' in warning:
                print(f"   Typical Error: \"{warning['error_example']}\"")
            print(f"   Install from: {warning['install_url']}")
            print()
    
    # Ask user to continue or exit
    if not report['all_critical_met']:
        print_error("CRITICAL COMPONENTS MISSING!")
        
        # Build list of missing critical components
        missing_critical = []
        if not report['extractor_installed']:
            missing_critical.append("KFXKeyExtractor28.exe")
        if not report['kindle_installed']:
            missing_critical.append("Kindle for PC")
        
        if len(missing_critical) == 1:
            print(f"The script cannot proceed without {missing_critical[0]}.")
        else:
            print(f"The script cannot proceed without: {', '.join(missing_critical)}.")
        print()
        return False
    
    if report['warnings']:
        print_warn("Some components are missing. Script can continue but will most likely fail.")
        print()
        print("Options:")
        print("  [C] Continue - Proceed with warnings (will most likely fail during import/conversion)")
        print("  [Q] Quit - Exit and install missing components")
        print()
        
        while True:
            choice = input("Your choice (C/Q) [Q]: ").strip().upper()
            if choice == '':
                choice = 'Q' #Default to Quit Script
            if choice == 'C':
                print()
                print_ok("Continuing with warnings...")
                print()
                return True
            elif choice == 'Q':
                print()
                print_warn("Script cancelled - please install missing components")
                print()
                return False
            else:
                print_error("Invalid choice. Please enter C or Q.")
    
    # All components present
    print_ok("All components validated successfully!")
    print()
    return True

# ============================================================================
# UNIFIED CONFIGURATION FUNCTIONS
# ============================================================================

def load_config():
    """
    Load unified configuration from JSON file
    Checks both script directory and fallback AppData location
    Returns dict or None if file doesn't exist
    """
    script_dir = os.path.dirname(os.path.abspath(__file__))
    user_home = os.path.expanduser("~")
    
    # Check script directory first
    script_config_path = os.path.join(script_dir, CONFIG_FILE)
    
    # Check fallback location
    fallback_base = os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    fallback_config_path = os.path.join(fallback_base, CONFIG_FILE)
    
    # Priority 1: Check fallback location first (most likely if script is in read-only location)
    if os.path.exists(fallback_config_path):
        try:
            with open(fallback_config_path, 'r') as f:
                config = json.load(f)
            return config
        except Exception as e:
            print_warn(f"Failed to load configuration from AppData: {e}")
    
    # Priority 2: Check script directory
    if os.path.exists(script_config_path):
        try:
            with open(script_config_path, 'r') as f:
                config = json.load(f)
            return config
        except Exception as e:
            print_warn(f"Failed to load configuration from script directory: {e}")
    
    return None

def save_config(config):
    """
    Save unified configuration to JSON file
    Uses the validated working directory (either script_dir or fallback)
    Returns True if successful, False otherwise
    """
    script_dir = os.path.dirname(os.path.abspath(__file__))
    user_home = os.path.expanduser("~")
    
    # Check if script_dir is writable first
    can_write, error = check_write_permissions(script_dir)
    
    if can_write:
        # Script dir is writable - use it
        config_path = os.path.join(script_dir, CONFIG_FILE)
    else:
        # Script dir not writable - use fallback
        fallback_base = os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
        config_path = os.path.join(fallback_base, CONFIG_FILE)
    
    try:
        # Ensure directory exists
        os.makedirs(os.path.dirname(config_path), exist_ok=True)
        
        # Add script version and timestamp
        config['script_version'] = SCRIPT_VERSION  # type: ignore[index]
        config['last_updated'] = datetime.now().isoformat()  # type: ignore[index]
        
        with open(config_path, 'w') as f:
            json.dump(config, f, indent=2)
        
        print_ok(f"Configuration saved to: {config_path}")
        return True
    except Exception as e:
        print_error(f"Failed to save configuration: {e}")
        return False

def check_config_version(config):
    """
    Check if config version matches current script version
    Returns: (is_valid: bool, config_version: str, current_version: str)
    """
    config_version = config.get('script_version', 'Unknown')
    is_valid = (config_version == SCRIPT_VERSION)
    return is_valid, config_version, SCRIPT_VERSION

def delete_config():
    """Delete configuration file"""
    script_dir = os.path.dirname(os.path.abspath(__file__))
    config_path = os.path.join(script_dir, CONFIG_FILE)
    
    try:
        if os.path.exists(config_path):
            os.remove(config_path)
            print_ok("Configuration deleted")
            return True
        return False
    except Exception as e:
        print_error(f"Failed to delete configuration: {e}")
        return False

def display_config_summary(config):
    """Display configuration summary in table format"""
    print_step("Current Configuration:")
    print()
    
    # Table header (wider to accommodate long paths - 72 chars for value column)
    print("┌─────────────────────────────┬──────────────────────────────────────────────────────────────────────────┐")
    print("│ Setting                     │ Value                                                                    │")
    print("├─────────────────────────────┼──────────────────────────────────────────────────────────────────────────┤")
    
    # Kindle Content Path
    content_path = config.get('kindle_content_path', 'Not set')
    if len(content_path) > 72:
        content_path = content_path[:69] + "..."
    print(f"│ Kindle Content Path         │ {content_path:<72} │")
    
    # Hide Sensitive Info
    hide_sensitive = "Yes" if config.get('hide_sensitive_info', False) else "No"
    print(f"│ Hide Sensitive Info         │ {hide_sensitive:<72} │")
    
    # Fetch Book Titles
    fetch_titles = "Yes" if config.get('fetch_book_titles', False) else "No"
    print(f"│ Fetch Book Titles           │ {fetch_titles:<72} │")
    
    # Clear Screen Between Phases
    clear_screen = "Yes" if config.get('clear_screen_between_phases', True) else "No"
    print(f"│ Clear Screen Between Phases │ {clear_screen:<72} │")
    
    # Skip Phase Pauses
    skip_pauses = "Yes" if config.get('skip_phase_pauses', False) else "No"
    print(f"│ Skip Phase Pauses           │ {skip_pauses:<72} │")
    
    # Auto-Launch Kindle
    auto_launch = "Yes" if config.get('auto_launch_kindle', False) else "No"
    print(f"│ Auto-Launch Kindle          │ {auto_launch:<72} │")
    
    # Raw Debug Logs
    raw_logs = "Yes" if config.get('enable_raw_logs', False) else "No"
    print(f"│ Raw Debug Logs              │ {raw_logs:<72} │")
    
    # Calibre Import section
    if 'calibre_import' in config:
        cal = config['calibre_import']
        calibre_status = "Enabled" if cal.get('enabled') else "Disabled"
        print(f"│ Calibre Import              │ {calibre_status:<72} │")
        
        if cal.get('enabled'):
            # Library Path with book count
            lib_path = cal.get('library_path', 'Not set')
            
            # Try to get book count
            book_count = get_library_book_count(lib_path) if lib_path != 'Not set' else None
            
            # Format display with book count
            if book_count is not None:
                display_path = f"{lib_path} ({book_count} books)"
            else:
                display_path = f"{lib_path} (Unknown books)"
            
            if len(display_path) > 72:
                display_path = display_path[:69] + "..."
            print(f"│   Library Path              │ {display_path:<72} │")
            
            # Convert to EPUB
            convert_epub = "Yes" if cal.get('convert_to_epub', False) else "No"
            print(f"│   Convert to EPUB           │ {convert_epub:<72} │")
            
            # KFX-ZIP Handling
            kfx_mode = cal.get('kfx_zip_mode', 'convert_all')
            kfx_mode_display = {
                'convert_all': 'Convert All (including .kfx-zip)',
                'skip_kfx_zip': 'Skip .kfx-zip files',
                'convert_regular_only': 'Convert regular .kfx only'
            }.get(kfx_mode, kfx_mode)
            print(f"│   KFX-ZIP Handling          │ {kfx_mode_display:<72} │")
            
            # Source Management
            source_mgmt = cal.get('source_file_management', 'keep_both')
            source_mgmt_display = {
                'keep_both': 'Keep Both (Source + EPUB)',
                'delete_source': 'Delete Source after conversion',
                'delete_kfx_zip_only': 'Delete .kfx-zip only'
            }.get(source_mgmt, source_mgmt)
            print(f"│   Source Management         │ {source_mgmt_display:<72} │")
    else:
        print(f"│ Calibre Import              │ {'Not configured':<72} │")
    
    # Table footer
    print("└─────────────────────────────┴──────────────────────────────────────────────────────────────────────────┘")
    print()

def prompt_config_action_with_timer(config):
    """Display config and timer with interrupt capability"""
    print_step("Configuration Found")
    print("--------------------------------------------------")
    display_config_summary(config)
    
    # Validate Calibre library path if Calibre import is enabled
    if 'calibre_import' in config and config['calibre_import'].get('enabled', False):
        lib_path = config['calibre_import'].get('library_path')
        if lib_path:
            print_step("Validating saved Calibre library path...")
            valid, error, book_count = verify_library_path(lib_path)
            
            if not valid:
                print()
                print_error("SAVED LIBRARY PATH IS INVALID!")
                print(f"  Path: {lib_path}")
                print(f"  Error: {error}")
                print()
                print_warn("The saved Calibre library path is no longer valid.")
                print("This could happen if:")
                print("  - Library was deleted or moved")
                print("  - Network drive is disconnected")
                print("  - Running on different machine")
                print("  - metadata.db was removed or corrupted")
                print()
                print("Options:")
                print("  [R] Reconfigure - Update library path")
                print("  [D] Disable - Turn off Calibre import")
                print("  [Q] Quit - Exit script")
                print()
                
                while True:
                    choice = input("Your choice (R/D/Q): ").strip().upper()
                    if choice == 'R':
                        print()
                        return 'reconfigure'
                    elif choice == 'D':
                        print()
                        print_step("Disabling Calibre import...")
                        config['calibre_import']['enabled'] = False
                        save_config(config)
                        print_ok("Calibre import disabled")
                        print()
                        return 'use'
                    elif choice == 'Q':
                        print()
                        return 'quit'
                    else:
                        print_error("Invalid choice. Please enter R, D, or Q.")
            else:
                if book_count is not None:
                    print_ok(f"Library path validated: {book_count} books found")
                else:
                    print_ok("Library path validated")
                print()
    
    # Check if pauses should be skipped to adjust countdown time
    skip_pauses = config.get('skip_phase_pauses', False)
    countdown_seconds = 3 if skip_pauses else 5
    
    print(f"Press any key to show options, or wait {countdown_seconds} seconds to use saved configuration...")
    print()
    
    timer_cancelled = threading.Event()
    user_interrupted = threading.Event()
    countdown_active = True
    
    def countdown_timer():
        nonlocal countdown_active
        for i in range(countdown_seconds, 0, -1):
            if timer_cancelled.is_set() or user_interrupted.is_set():
                countdown_active = False
                return
            print(f"\rCountdown: {i} seconds... ", end='', flush=True)
            time.sleep(1)
        countdown_active = False
        if not user_interrupted.is_set():
            print("\r" + " " * 50 + "\r", end='', flush=True)
            print_ok("Auto-proceeding with saved configuration")
            timer_cancelled.set()
    
    timer_thread = threading.Thread(target=countdown_timer, daemon=True)
    timer_thread.start()
    
    while countdown_active:
        if msvcrt.kbhit():
            msvcrt.getch()
            user_interrupted.set()
            timer_cancelled.set()
            print("\r" + " " * 50 + "\r", end='', flush=True)
            break
        time.sleep(0.05)
    
    if not user_interrupted.is_set():
        return 'use'
    
    print()
    print_step("Configuration Options:")
    print("  [U] Use saved configuration")
    print("  [R] Reconfigure settings")
    print("  [D] Delete saved config and reconfigure")
    print("  [Q] Quit script")
    print()
    
    while True:
        choice = input("Your choice (U/R/D/Q): ").strip().upper()
        if choice == 'U':
            print()
            return 'use'
        elif choice == 'R':
            print()
            return 'reconfigure'
        elif choice == 'D':
            print()
            delete_config()
            return 'reconfigure'
        elif choice == 'Q':
            print()
            return 'quit'
        else:
            print_error("Invalid choice. Please enter U, R, D, or Q.")

# ============================================================================
# PRE-FLIGHT CONFIGURATION WIZARD
# ============================================================================

def configure_pre_flight_wizard(user_home, TOTAL_STEPS):
    """
    Pre-Flight Configuration Wizard
    Comprehensive setup for all script options
    """
    
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print("This wizard will guide you through configuring all script options.")
    print("You can change these settings later by deleting the config file.")
    print()
    
    config = {
        'script_version': SCRIPT_VERSION,
        'last_updated': datetime.now().isoformat()
    }
    
    # 1. Kindle Content Path - Use get_kindle_content_path() for validation
    print_step(f"[1/{TOTAL_STEPS}] Kindle Content Directory")
    print("--------------------------------------------------")
    default_content = os.path.join(user_home, "Documents", "My Kindle Content")
    print()
    
    # Use get_kindle_content_path() which includes validation workflow
    # Disable timer in wizard - user should make deliberate choice during first-time config
    config['kindle_content_path'] = get_kindle_content_path(default_content, use_timer=False)
    
    print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    print()
    
    # 2. Hide Sensitive Information
    print_step(f"[2/{TOTAL_STEPS}] Privacy Settings")
    print("--------------------------------------------------")
    print("Hide sensitive information (DSN, tokens, keys) in console output?")
    print()
    
    while True:
        choice = input("Hide sensitive info? (Y/N) [Y]: ").strip().upper()
        if choice == '':
            choice = 'Y'  # Default to Yes
        if choice in ['Y', 'N']:
            config['hide_sensitive_info'] = (choice == 'Y')  # type: ignore[assignment]
            print()
            if choice == 'Y':
                print_ok("✓ Sensitive information will be hidden")
            else:
                print_ok("✓ Sensitive information will be shown")
            break
        print_error("Please enter Y or N")
    
    print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    print()
    
    # 3. Fetch Book Titles During Key Extraction
    print_step(f"[3/{TOTAL_STEPS}] Key Extraction Display Options")
    print("--------------------------------------------------")
    print("Fetch book titles from Amazon during key extraction?")
    print()
    print_warn("NOTE:")
    print("  - Fetching titles queries Amazon servers for each book")
    print("  - This will significantly slow down the extraction process")
    print("  - Titles are only used for display purposes during extraction")
    print("  - Recommended: No (faster extraction)")
    print()
    
    while True:
        choice = input("Fetch book titles during extraction? (Y/N) [N]: ").strip().upper()
        if choice == '':
            choice = 'N'  # Default to No (faster)
        if choice in ['Y', 'N']:
            config['fetch_book_titles'] = (choice == 'Y')  # type: ignore[assignment]
            print()
            if choice == 'Y':
                print_ok("✓ Book titles will be fetched (slower extraction)")
            else:
                print_ok("✓ Book titles will NOT be fetched (faster extraction)")
            break
        print_error("Please enter Y or N")
    
    print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    fetch_status = "Yes" if config['fetch_book_titles'] else "No"
    print_ok(f"✓ [3/{TOTAL_STEPS}] Fetch Book Titles: {fetch_status}")
    print()
    
    # 4. Clear Screen Between Phases
    print_step(f"[4/{TOTAL_STEPS}] Display Options")
    print("--------------------------------------------------")
    print("Clear screen between each phase for cleaner output?")
    print()
    print_warn("NOTE:")
    print("  - Clearing screen provides a cleaner, less cluttered view")
    print("  - Each phase will start with a fresh screen")
    print("  - Phase summaries will still be displayed before clearing")
    print("  - Recommended: Yes (cleaner output)")
    print()
    
    while True:
        choice = input("Clear screen between phases? (Y/N) [N]: ").strip().upper()
        if choice == '':
            choice = 'N'  # Default to No
        if choice in ['Y', 'N']:
            config['clear_screen_between_phases'] = (choice == 'Y')  # type: ignore[assignment]
            print()
            if choice == 'Y':
                print_ok("✓ Screen will be cleared between phases")
            else:
                print_ok("✓ Screen will NOT be cleared between phases")
            break
        print_error("Please enter Y or N")
    
    print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    fetch_status = "Yes" if config['fetch_book_titles'] else "No"
    print_ok(f"✓ [3/{TOTAL_STEPS}] Fetch Book Titles: {fetch_status}")
    clear_status = "Yes" if config['clear_screen_between_phases'] else "No"
    print_ok(f"✓ [4/{TOTAL_STEPS}] Clear Screen Between Phases: {clear_status}")
    print()
    
    # 5. Skip Phase Pauses
    print_step(f"[5/{TOTAL_STEPS}] Phase Pause Settings")
    print("--------------------------------------------------")
    print("Skip countdown pauses between phases for faster execution?")
    print()
    print_warn("NOTE:")
    print("  - Pauses allow you to review phase summaries before continuing")
    print("  - Skipping pauses makes the script run faster without interruption")
    print("  - Initial config review pause will be reduced to 3 seconds (from 10)")
    print("  - Phase summaries will still be displayed even if pauses are skipped")
    print("  - Recommended: No (keep pauses for better visibility)")
    print()
    
    while True:
        choice = input("Skip phase pauses? (Y/N) [N]: ").strip().upper()
        if choice == '':
            choice = 'N'  # Default to No (keep pauses)
        if choice in ['Y', 'N']:
            config['skip_phase_pauses'] = (choice == 'Y')  # type: ignore[assignment]
            print()
            if choice == 'Y':
                print_ok("✓ Phase pauses will be skipped (faster execution)")
            else:
                print_ok("✓ Phase pauses will be shown (better visibility)")
            break
        print_error("Please enter Y or N")
    
    print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    fetch_status = "Yes" if config['fetch_book_titles'] else "No"
    print_ok(f"✓ [3/{TOTAL_STEPS}] Fetch Book Titles: {fetch_status}")
    clear_status = "Yes" if config['clear_screen_between_phases'] else "No"
    print_ok(f"✓ [4/{TOTAL_STEPS}] Clear Screen Between Phases: {clear_status}")
    skip_pauses_status = "Yes" if config['skip_phase_pauses'] else "No"
    print_ok(f"✓ [5/{TOTAL_STEPS}] Skip Phase Pauses: {skip_pauses_status}")
    print()

    # 6. Auto-Launch Kindle
    print_step(f"[6/{TOTAL_STEPS}] Auto-Launch Kindle")
    print("--------------------------------------------------")
    print("Would you like to automatically launch Kindle.exe?")
    print("(Script will wait for Kindle to close before scanning for books)")
    print()
    
    while True:
        choice = input("Enable Auto-Launch Kindle? (Y/N) [N]: ").strip().upper()
        if choice == '':
            choice = 'N'  # Default to No
        if choice in ['Y', 'N']:
            config['auto_launch_kindle'] = (choice == 'Y')  # type: ignore[assignment]
            print()
            break
        else:
            print_error("Please enter Y or N")
            print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    fetch_status = "Yes" if config['fetch_book_titles'] else "No"
    print_ok(f"✓ [3/{TOTAL_STEPS}] Fetch Book Titles: {fetch_status}")
    clear_status = "Yes" if config['clear_screen_between_phases'] else "No"
    print_ok(f"✓ [4/{TOTAL_STEPS}] Clear Screen Between Phases: {clear_status}")
    skip_pauses_status = "Yes" if config['skip_phase_pauses'] else "No"
    print_ok(f"✓ [5/{TOTAL_STEPS}] Skip Phase Pauses: {skip_pauses_status}")
    auto_launch_status = "Yes" if config['auto_launch_kindle'] else "No"
    print_ok(f"✓ [6/{TOTAL_STEPS}] Auto-Launch Kindle: {auto_launch_status}")
    print()

    # 7. Raw Debug Logs
    print_step(f"[7/{TOTAL_STEPS}] Raw Debug Logs")
    print("--------------------------------------------------")
    print("Enable raw debug logs for complete subprocess output?")
    print()
    print_warn("NOTE:")
    print("  - Raw logs capture ALL stdout/stderr from subprocesses")
    print("  - Useful for debugging when complete output is needed")
    print("  - Console display will still only show errors")
    print("  - Logs are auto-rotated (keeps 10 most recent)")
    print("  - Stored in Logs/extraction_logs/, import_logs/, conversion_logs/")
    print("  - Recommended: No (only enable when debugging)")
    print()
    
    while True:
        choice = input("Enable raw debug logs? (Y/N) [Y]: ").strip().upper()
        if choice == '':
            choice = 'Y'  # Default to Yes
        if choice in ['Y', 'N']:
            config['enable_raw_logs'] = (choice == 'Y')  # type: ignore[assignment]
            print()
            if choice == 'Y':
                print_ok("✓ Raw debug logs will be enabled")
            else:
                print_ok("✓ Raw debug logs will be disabled")
            break
        print_error("Please enter Y or N")
    
    print()
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    fetch_status = "Yes" if config['fetch_book_titles'] else "No"
    print_ok(f"✓ [3/{TOTAL_STEPS}] Fetch Book Titles: {fetch_status}")
    clear_status = "Yes" if config['clear_screen_between_phases'] else "No"
    print_ok(f"✓ [4/{TOTAL_STEPS}] Clear Screen Between Phases: {clear_status}")
    skip_pauses_status = "Yes" if config['skip_phase_pauses'] else "No"
    print_ok(f"✓ [5/{TOTAL_STEPS}] Skip Phase Pauses: {skip_pauses_status}")
    auto_launch_status = "Yes" if config['auto_launch_kindle'] else "No"
    print_ok(f"✓ [6/{TOTAL_STEPS}] Auto-Launch Kindle: {auto_launch_status}")
    enable_raw_logs = "Yes" if config['enable_raw_logs'] else "No"
    print_ok(f"✓ [7/{TOTAL_STEPS}] Enable Raw Debug Logs: {enable_raw_logs}")
    print()
    
    # 8. Calibre Import Settings
    print_step(f"[8/{TOTAL_STEPS}] Calibre Auto-Import")
    print("--------------------------------------------------")
    print("Enable automatic import of DeDRMed ebooks to Calibre?")
    print("(You can configure this later if you skip now)")
    print()
    
    while True:
        choice = input("Configure Calibre import now? (Y/N) [Y]: ").strip().upper()
        if choice == '':
            choice = 'Y'  # Default to Yes
        if choice in ['Y', 'N']:
            break
        print_error("Please enter Y or N")
    
    if choice == 'N':
        config['calibre_import'] = {'enabled': False}  # type: ignore[assignment]
        print()
        print_ok("✓ Calibre auto-import disabled")
        print()
        print()
    else:
        print()
        calibre_config = prompt_calibre_import_settings()
        if calibre_config:
            config['calibre_import'] = calibre_config  # type: ignore[assignment]
        else:
            config['calibre_import'] = {'enabled': False}  # type: ignore[assignment]
    
    # Clear console before next question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok(f"✓ [1/{TOTAL_STEPS}] Kindle Content Path: {config['kindle_content_path']}")
    hide_status = "Yes" if config['hide_sensitive_info'] else "No"
    print_ok(f"✓ [2/{TOTAL_STEPS}] Hide Sensitive Info: {hide_status}")
    fetch_status = "Yes" if config['fetch_book_titles'] else "No"
    print_ok(f"✓ [3/{TOTAL_STEPS}] Fetch Book Titles: {fetch_status}")
    clear_status = "Yes" if config['clear_screen_between_phases'] else "No"
    print_ok(f"✓ [4/{TOTAL_STEPS}] Clear Screen Between Phases: {clear_status}")
    skip_pauses_status = "Yes" if config['skip_phase_pauses'] else "No"
    print_ok(f"✓ [5/{TOTAL_STEPS}] Skip Phase Pauses: {skip_pauses_status}")
    auto_launch_status = "Yes" if config['auto_launch_kindle'] else "No"
    print_ok(f"✓ [6/{TOTAL_STEPS}] Auto-Launch Kindle: {auto_launch_status}")
    calibre_status = "Enabled" if config['calibre_import'].get('enabled') else "Disabled"
    print_ok(f"✓ [8/{TOTAL_STEPS}] Calibre Auto-Import: {calibre_status}")
    print()
    
    # Display final configuration review
    while True:
        # Clear console before showing review
        os.system('cls')
        print()
        print_banner_and_version()
        print("=" * 70)
        print_step("CONFIGURATION REVIEW")
        print("=" * 70)
        print()
        print("Please review your configuration before saving:")
        print()
        display_config_summary(config)
        
        print("Options:")
        print("  [Y] Yes, save and continue (recommended)")
        print("  [R] Reconfigure - Start over")
        print("  [Q] Quit without saving")
        print()
        
        review_choice = input("Your choice (Y/R/Q) [Y]: ").strip().upper()
        if review_choice == '':
            review_choice = 'Y'
        
        if review_choice == 'Y':
            # Save configuration
            print()
            print_step("Saving Configuration")
            print("--------------------------------------------------")
            save_config(config)
            print()
            
            print("=" * 70)
            print_done("PRE-FLIGHT CONFIGURATION COMPLETE")
            print("=" * 70)
            print()
            
            return config
        elif review_choice == 'R':
            print()
            print_step("Restarting configuration wizard...")
            print()
            # Recursively call the wizard to start over
            return configure_pre_flight_wizard(user_home, TOTAL_STEPS)
        elif review_choice == 'Q':
            print()
            print_warn("Configuration cancelled - exiting without saving")
            sys.exit(0)
        else:
            print_error("Invalid choice. Please enter Y, R, or Q.")

# ============================================================================
# CALIBRE AUTO-IMPORT FUNCTIONS
# ============================================================================

def load_calibre_config():
    """
    Load saved Calibre import configuration from JSON file (legacy support)
    Returns dict or None if file doesn't exist
    """
    # Try new unified config first
    config = load_config()
    if config and 'calibre' in config:
        return config['calibre']
    
    # Fall back to old calibre_import_config.json
    script_dir = os.path.dirname(os.path.abspath(__file__))
    config_path = os.path.join(script_dir, "calibre_import_config.json")
    
    if os.path.exists(config_path):
        try:
            with open(config_path, 'r') as f:
                config = json.load(f)
            return config
        except Exception as e:
            print_warn(f"Failed to load saved configuration: {e}")
            return None
    return None

def save_calibre_config(config):
    """
    Save Calibre import configuration to JSON file (legacy support)
    Returns True if successful, False otherwise
    """
    script_dir = os.path.dirname(os.path.abspath(__file__))
    config_path = os.path.join(script_dir, "calibre_import_config.json")
    
    try:
        # Add timestamp
        config['last_updated'] = datetime.now().isoformat()
        
        with open(config_path, 'w') as f:
            json.dump(config, f, indent=2)
        
        print_ok(f"Configuration saved to: {config_path}")
        return True
    except Exception as e:
        print_error(f"Failed to save configuration: {e}")
        return False

def display_saved_config(config):
    """
    Display saved configuration
    """
    print_step("Saved Calibre Import Configuration:")
    print(f"  Library Path: {config.get('library_path', 'Not set')}")
    print()

def prompt_calibre_with_timer(saved_config):
    """
    Display saved config and 5-second timer with interrupt capability
    Returns: 'use' | 'reconfigure' | 'skip' | 'delete'
    """
    print_step("Calibre Auto-Import Configuration Found")
    print("--------------------------------------------------")
    display_saved_config(saved_config)
    
    print("Press any key to show options, or wait 5 seconds to use saved configuration...")
    print()
    
    # Shared state for timer and input
    timer_cancelled = threading.Event()
    user_interrupted = threading.Event()
    countdown_active = True
    
    def countdown_timer():
        nonlocal countdown_active
        for i in range(5, 0, -1):
            if timer_cancelled.is_set() or user_interrupted.is_set():
                countdown_active = False
                return
            print(f"\rCountdown: {i} seconds... ", end='', flush=True)
            time.sleep(1)
        countdown_active = False
        if not user_interrupted.is_set():
            print("\r" + " " * 50 + "\r", end='', flush=True)
            print_ok("Auto-proceeding with saved configuration")
            timer_cancelled.set()
    
    # Start countdown timer
    timer_thread = threading.Thread(target=countdown_timer, daemon=True)
    timer_thread.start()
    
    # Wait for user input or timer expiry
    while countdown_active:
        if msvcrt.kbhit():
            # User pressed a key - cancel timer and show menu
            msvcrt.getch()  # Consume the keypress
            user_interrupted.set()
            timer_cancelled.set()
            print("\r" + " " * 50 + "\r", end='', flush=True)
            break
        time.sleep(0.05)
    
    # If timer expired without interruption, use saved config
    if not user_interrupted.is_set():
        return 'use'
    
    # Show menu
    print()
    print_step("Configuration Options:")
    print("  [U] Use saved configuration")
    print("  [R] Reconfigure settings")
    print("  [S] Skip auto-import this time")
    print("  [D] Delete saved config and skip")
    print()
    
    while True:
        choice = input("Your choice (U/R/S/D): ").strip().upper()
        if choice == 'U':
            print()
            return 'use'
        elif choice == 'R':
            print()
            return 'reconfigure'
        elif choice == 'S':
            print()
            return 'skip'
        elif choice == 'D':
            print()
            return 'delete'
        else:
            print_error("Invalid choice. Please enter U, R, S, or D.")

def get_last_calibre_library():
    """
    Read last library from Calibre's global.py.json
    Returns: library_path or None if not found
    """
    try:
        config_file = os.path.join(
            os.path.expanduser("~"),
            "AppData", "Roaming", "calibre", "global.py.json"
        )
        
        if not os.path.exists(config_file):
            return None
        
        with open(config_file, 'r') as f:
            config = json.load(f)
        
        return config.get('library_path')
        
    except Exception as e:
        print_warn(f"Could not read Calibre config: {e}")
        return None

def get_library_book_count(library_path):
    """
    Get the total number of books in a Calibre library
    Returns: book_count (int) or None if query fails
    """
    import sqlite3
    
    try:
        db_path = os.path.join(library_path, "metadata.db")
        conn = sqlite3.connect(db_path, timeout=5)
        cursor = conn.cursor()
        cursor.execute("SELECT COUNT(*) FROM books")
        count = cursor.fetchone()[0]
        conn.close()
        return count
    except Exception:
        return None  # Return None on any error

def reset_calibredb_book_id(library_path):
    """
    Silently reset Calibre's book ID counter if library has zero books.
    
    Args:
        library_path: Path to Calibre library directory
        
    Returns:
        bool: True if reset successful, False otherwise (silent operation)
    """
    import sqlite3
    
    try:
        # Check book count - silent abort if not 0
        book_count = get_library_book_count(library_path)
        if book_count is None or book_count > 0:
            return False
        
        # Perform reset silently
        db_path = os.path.join(library_path, "metadata.db")
        if not os.path.exists(db_path):
            return False
        
        conn = sqlite3.connect(db_path)
        cur = conn.cursor()
        cur.execute("DELETE FROM books;")
        cur.execute("DELETE FROM sqlite_sequence WHERE name='books';")
        conn.commit()
        conn.close()
        
        return True
    except:
        return False  # Silent failure

def verify_library_path(library_path):
    """
    Verify library path exists and contains metadata.db
    Returns: (valid: bool, error_message: str, book_count: int or None)
    """
    if not os.path.exists(library_path):
        return False, "Library path does not exist", None
    
    metadata_db = os.path.join(library_path, "metadata.db")
    if not os.path.exists(metadata_db):
        return False, "Not a valid Calibre library (metadata.db not found)", None
    
    # Get book count
    book_count = get_library_book_count(library_path)
    
    return True, "", book_count

def prompt_manual_library_path():
    """
    Prompt user to manually enter library path
    Returns: validated library path
    """
    print_step("Enter Calibre Library Path")
    print("This is the folder containing metadata.db")
    print("Example: D:\\OneDrive\\Calibre-Libraries\\Temp-downloads")
    print()
    
    while True:
        library_path = input("Library path: ").strip().strip('"').strip("'")
        
        if not library_path:
            print_error("Path cannot be empty")
            continue
        
        # Normalize path
        library_path = os.path.normpath(library_path)
        
        # Verify it's valid
        valid, error, book_count = verify_library_path(library_path)
        
        if valid:
            print_ok(f"Library path validated: {library_path}")
            if book_count is not None:
                print(f"  Books found: {book_count}")
            else:
                print(f"  Books found: Unknown")
            print()
            return library_path
        else:
            print_error(f"Invalid library path: {error}")
            print()
            retry = input("Try again? (Y/N): ").strip().upper()
            if retry != 'Y':
                raise Exception("Library path validation cancelled")

def get_and_confirm_library_path():
    """
    Get library path with fallback to manual entry
    Returns: validated library path
    """
    # Try to read last library from Calibre config
    last_library = get_last_calibre_library()
    
    if last_library:
        # Get book count for display
        book_count = get_library_book_count(last_library)
        if book_count is not None:
            print_ok(f"Last used Calibre library: {last_library} ({book_count} books)")
        else:
            print_ok(f"Last used Calibre library: {last_library}")
        print()
        choice = input("Use this library? (Y/N) [Y]: ").strip().upper()
        if choice == '':
            choice = 'Y'  # Default to Yes
        
        if choice == 'Y':
            # Verify it's valid
            valid, error, book_count = verify_library_path(last_library)
            if valid:
                if book_count is not None:
                    print_ok(f"Library validated: {book_count} books found")
                else:
                    print_ok(f"Library validated: Unknown book count")
                print()
                return last_library
            else:
                print_warn(f"Library path invalid: {error}")
                print_warn("Please enter a valid library path")
                print()
    else:
        print_warn("Could not find Calibre configuration")
        print_warn("Please enter your Calibre library path manually")
        print()
    
    # Fallback: Manual entry
    return prompt_manual_library_path()

def prompt_calibre_import_settings():
    """
    Prompt user for Calibre import settings
    Returns: dict with configuration or None if user declines
    """
    # Track configuration progress
    config_progress = {}
    
    # Initial screen
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_step("[8/8] Calibre Auto-Import")
    print("--------------------------------------------------")
    print()
    
    # Check if Calibre is running
    if not is_calibre_running():
        print_ok("Calibre not detected as Running - proceeding automatically")
        print()
    else:
        # Important: Calibre must be closed to read configuration files
        print_warn("IMPORTANT: Please close Calibre now before proceeding")
        print()
        print("During configuration, the script needs to read Calibre's configuration")
        print("files to detect your last used library path. This requires Calibre to")
        print("be completely closed.")
        print()
        
        print_step("Waiting for Calibre to close...")
        print("  (Script will automatically continue when Calibre closes)")
        print()
        
        # Auto-detection loop - wait until Calibre closes
        while is_calibre_running():
            time.sleep(1)  # Check every second
        
        print_ok("Calibre has closed - continuing with configuration...")
        print()
    
    # Get and confirm library path
    library_path = get_and_confirm_library_path()
    config_progress['library_path'] = library_path
    
    # Get book count for display
    book_count = get_library_book_count(library_path)
    config_progress['book_count'] = book_count
    
    # Clear screen and show progress before EPUB conversion question
    os.system('cls')
    print()
    print_banner_and_version()
    print("=" * 70)
    print_step("PRE-FLIGHT CONFIGURATION WIZARD")
    print("=" * 70)
    print()
    print_ok("✓ [8/8] Calibre Auto-Import")
    if book_count is not None:
        print(f"  ✓ Library Path: {config_progress['library_path']} ({book_count} books)")
    else:
        print(f"  ✓ Library Path: {config_progress['library_path']}")
    print()
    
    # Ask about Imported eBook to EPUB conversion
    print_step("Imported eBook to EPUB Conversion")
    print("--------------------------------------------------")
    print()
    print("After importing ebooks, they can be automatically converted to EPUB format.")
    print()
    print_warn("NOTES:")
    print("  - Conversion uses ebook-convert command")
    print("  - EPUB format will be merged into existing book records")
    print("  - This may take additional time depending on book count")
    print()
    
    while True:
        convert_choice = input("Convert imported eBooks to EPUB? (Y/N) [Y]: ").strip().upper()
        if convert_choice == '':
            convert_choice = 'Y'  # Default to Yes
        if convert_choice in ['Y', 'N']:
            break
        print_error("Please enter Y or N")
    
    config_progress['convert_to_epub'] = (convert_choice == 'Y')
    
    # Ask about skipping .kfx-zip files (only if conversion is enabled)
    kfx_zip_mode = 'convert_all'  # Default
    source_file_management = 'keep_both'  # Default
    
    if convert_choice == 'Y':
        # Clear screen and show progress before KFX-ZIP handling question
        os.system('cls')
        print()
        print_banner_and_version()
        print("=" * 70)
        print_step("PRE-FLIGHT CONFIGURATION WIZARD")
        print("=" * 70)
        print()
        print_ok("✓ [8/8] Calibre Auto-Import")
        print(f"  ✓ Library Path: {config_progress['library_path']}")
        convert_status = "Yes" if config_progress['convert_to_epub'] else "No"
        print(f"  ✓ Convert to EPUB: {convert_status}")
        print()
        
        print_step("KFX-ZIP File Handling")
        print("--------------------------------------------------")
        print()
        print("Files with .kfx-zip extension usually indicate DRM protection failure.")
        print("These files rarely convert successfully to EPUB.")
        print()
        print("Conversion modes:")
        print("  [A] Convert All - Attempt to convert all files including .kfx-zip (recommended)")
        print("  [S] Skip KFX-ZIP - Skip .kfx-zip files, convert regular .kfx files only")
        print()
        
        while True:
            mode_choice = input("Your choice (A/S) [A]: ").strip().upper()
            if mode_choice == '':
                mode_choice = 'A'  # Default to recommended option
            if mode_choice == 'A':
                kfx_zip_mode = 'convert_all'
                print()
                print_ok("Will attempt to convert all files including .kfx-zip")
                break
            elif mode_choice == 'S':
                kfx_zip_mode = 'skip_kfx_zip'
                print()
                print_ok("Will skip .kfx-zip files during conversion")
                break
            else:
                print_error("Invalid choice. Please enter A or S.")
        
        config_progress['kfx_zip_mode'] = kfx_zip_mode
        
        # Clear screen and show progress before source management question
        os.system('cls')
        print()
        print_banner_and_version()
        print("=" * 70)
        print_step("PRE-FLIGHT CONFIGURATION WIZARD")
        print("=" * 70)
        print()
        print_ok("✓ [8/8] Calibre Auto-Import")
        print(f"  ✓ Library Path: {config_progress['library_path']}")
        convert_status = "Yes" if config_progress['convert_to_epub'] else "No"
        print(f"  ✓ Convert to EPUB: {convert_status}")
        kfx_mode_display = "Convert All" if config_progress['kfx_zip_mode'] == 'convert_all' else "Skip KFX-ZIP"
        print(f"  ✓ KFX-ZIP Handling: {kfx_mode_display}")
        print()
        
        # Ask about source file management
        print_step("Source File Management")
        print("--------------------------------------------------")
        print()
        print("After successful EPUB conversion, what should happen to the source format files?")
        print()
        print("Options:")
        print("  [K] Keep Both - Preserve both source format and EPUB formats")
        print("  [D] Delete Source - Remove source format (KFX/AZW3/AZW) after successful EPUB conversion (recommended)")
        print("  [S] Smart Cleanup - Delete only .kfx-zip files, keep other formats")
        print()
        
        while True:
            choice = input("Your choice (K/D/S) [D]: ").strip().upper()
            if choice == '':
                choice = 'D'  # Default to recommended option
            if choice == 'K':
                print()
                print_ok("Will keep both source format and EPUB formats")
                source_file_management = 'keep_both'
                break
            elif choice == 'D':
                print()
                print_warn("Will delete source format after successful EPUB conversion")
                source_file_management = 'delete_source'
                break
            elif choice == 'S':
                print()
                print_ok("Will delete only .kfx-zip files, keeping other formats")
                source_file_management = 'delete_kfx_zip_only'
                break
            else:
                print_error("Invalid choice. Please enter K, D, or S.")
        
        print()
    
    # Build configuration
    config = {
        'enabled': True,
        'library_path': library_path,
        'convert_to_epub': convert_choice == 'Y',
        'kfx_zip_mode': kfx_zip_mode if convert_choice == 'Y' else 'convert_all',
        'source_file_management': source_file_management
    }
    
    return config

def cleanup_kfx_zip_books(library_path):
    """
    Query and optionally remove KFX-ZIP format books with user confirmation
    Returns: (removed_count, user_cancelled)
    """
    import sqlite3
    
    # Query for KFX-ZIP books with names
    db_path = os.path.join(library_path, "metadata.db")
    kfx_zip_books = []  # List of tuples: (book_id, book_name)
    
    try:
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        
        # Query: SELECT book, name FROM data WHERE format = 'KFX-ZIP'
        cursor.execute("SELECT book, name FROM data WHERE format = 'KFX-ZIP'")
        rows = cursor.fetchall()
        
        for row in rows:
            book_id = str(row[0])
            book_name = row[1]
            kfx_zip_books.append((book_id, book_name))
        
        conn.close()
        
    except Exception as e:
        print_error(f"Failed to query database: {e}")
        return 0, True
    
    if not kfx_zip_books:
        # No KFX-ZIP books found - silent success
        return 0, False
    
    # Show user-friendly list
    print()
    print_warn(f"Found {len(kfx_zip_books)} book(s) with KFX-ZIP format in database")
    print()
    print("These books may be DRM-protected versions that failed to decrypt.")
    print("Removing them allows clean import of newly decrypted versions.")
    print()
    print_warn("IMPORTANT: These KFX-ZIP books may NOT be related to current import!")
    print()
    print("Books to remove:")
    for book_id, book_name in kfx_zip_books:
        print(f"  [{book_id}] {book_name}")
    print()
    
    # User confirmation
    print("Options:")
    print("  [C] Continue - Remove these KFX-ZIP books and import (recommended)")
    print("  [S] Skip - Keep existing books, import with duplicates")
    print()
    
    while True:
        choice = input("Your choice (C/S) [C]: ").strip().upper()
        if choice == '':
            choice = 'C'  # Default to recommended option
        if choice == 'C':
            print()
            # Extract just the IDs for removal
            book_ids = [book_id for book_id, _ in kfx_zip_books]
            id_list = ','.join(book_ids)
            
            print_step(f"Removing {len(book_ids)} KFX-ZIP book(s)...")
            
            # Remove books using calibredb
            cmd = [
                'calibredb', 'remove',
                id_list,
                '--permanent',
                f'--library-path={library_path}'
            ]
            
            try:
                result = subprocess.run(cmd, capture_output=True, text=True, timeout=60)
                
                if result.returncode == 0:
                    print_ok(f"Successfully removed {len(book_ids)} book(s)")
                    print()
                    return len(book_ids), False
                else:
                    print_error(f"Failed to remove books: {result.stderr}")
                    print()
                    return 0, True
                    
            except Exception as e:
                print_error(f"Failed to remove books: {e}")
                print()
                return 0, True
                
        elif choice == 'S':
            print()
            print_warn("Skipping KFX-ZIP cleanup - will use duplicates mode")
            print()
            return 0, True
        else:
            print_error("Invalid choice. Please enter C or S.")

def find_all_azw_files(content_dir, exclude_asins=None):
    """
    Recursively find all .azw files in content directory
    Optionally exclude files matching ASINs that failed key extraction
    Returns: list of full file paths
    """
    azw_files = []
    exclude_asins = exclude_asins or []
    
    try:
        for root, dirs, files in os.walk(content_dir):
            for file in files:
                if file.lower().endswith('.azw'):
                    # Extract ASIN from filename (e.g., B00N17VVZC_EBOK.azw)
                    asin = file.split('_')[0] if '_' in file else file.replace('.azw', '')
                    
                    # Skip if this ASIN failed key extraction
                    if asin in exclude_asins:
                        continue
                    
                    full_path = os.path.join(root, file)
                    azw_files.append(full_path)
        
        return azw_files
        
    except Exception as e:
        print_error(f"Error scanning directory: {e}")
        return []

def import_single_book(book_path, library_path, use_duplicates=False, timeout_seconds=180, raw_log_path=None):
    """
    Import a single book to Calibre with timeout and timer display
    Returns: (success: bool, book_id: str or None, error_msg: str or None, timed_out: bool, already_exists: bool, book_title: str or None)
    """
    cmd = ['calibredb', 'add']
    
    if use_duplicates:
        cmd.append('-d')
    
    cmd.extend([
        '-1',  # One book per directory
        book_path,
        f'--library-path={library_path}'
    ])
    
    try:
        # Start subprocess with Popen for non-blocking execution
        process = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding='utf-8',
            errors='replace'
        )
        
        # Timer display thread
        timer_stopped = threading.Event()
        
        timer_thread = threading.Thread(
            target=display_progress_timer,
            args=("Importing", timeout_seconds, timer_stopped),
            daemon=True
        )
        timer_thread.start()
        
        # Wait for process with timeout
        try:
            stdout, stderr = process.communicate(timeout=timeout_seconds)
            result_returncode = process.returncode
            # Stop timer and clear the line on success
            timer_stopped.set()
            timer_thread.join(timeout=1)
            print("\r" + " " * 40 + "\r", end='', flush=True)
        except subprocess.TimeoutExpired:
            # Stop timer and clear the line on timeout
            timer_stopped.set()
            timer_thread.join(timeout=1)
            print("\r" + " " * 40 + "\r", end='', flush=True)
            process.kill()
            stdout, stderr = process.communicate()
            result_returncode = -1
        
        # Create a result-like object for compatibility
        class Result:
            def __init__(self, returncode, stdout, stderr):
                self.returncode = returncode
                self.stdout = stdout
                self.stderr = stderr
        
        result = Result(result_returncode, stdout, stderr)
        
        # Log to raw log if enabled
        if raw_log_path:
            book_name = os.path.basename(book_path)
            append_to_raw_log(
                raw_log_path,
                cmd,
                result.stdout,
                result.stderr,
                result.returncode,
                book_info=book_name
            )
        
        # Parse output for book ID
        if result.returncode == 0 and result.stdout:
            # Look for "Added book ids: 123"
            for line in result.stdout.split('\n'):
                if 'Added book ids:' in line:
                    ids_str = line.split('Added book ids:')[1].strip()
                    book_id = ids_str.split(',')[0].strip() if ids_str else None
                    return True, book_id, None, False, False, None
        
        # Check if book already exists in database
        error_msg = result.stderr if result.stderr else "Unknown error"
        already_exists = False
        book_title = None
        
        if "already exist in the database" in error_msg:
            already_exists = True
            # Extract book title from error message
            # Format: "The following books were not added as they already exist in the database:\n  Book Title"
            lines = error_msg.split('\n')
            for i, line in enumerate(lines):
                if "already exist in the database" in line:
                    # Next non-empty line should contain the book title
                    for j in range(i + 1, len(lines)):
                        title_line = lines[j].strip()
                        if title_line and not title_line.startswith('('):
                            book_title = title_line
                            break
                    break
        
        return False, None, error_msg, False, already_exists, book_title
        
    except subprocess.TimeoutExpired:
        # Log timeout to raw log if enabled
        if raw_log_path:
            book_name = os.path.basename(book_path)
            append_to_raw_log(
                raw_log_path,
                cmd,
                '',
                '',
                -1,  # Timeout exit code
                book_info=book_name
            )
        return False, None, f"Timeout after {timeout_seconds} seconds", True, False, None
        
    except Exception as e:
        return False, None, str(e), False, False, None

def write_import_log(library_path, results, azw_files, working_dir):
    """
    Write detailed import log to file
    Uses working_dir which respects fallback paths
    Returns: log file path
    """
    # Create logs directory with import subfolder
    logs_dir = os.path.join(working_dir, "Logs", "import_logs")
    os.makedirs(logs_dir, exist_ok=True)
    
    # Create timestamped log file
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    log_file = os.path.join(logs_dir, f"calibre_import_{timestamp}.log")
    
    try:
        with open(log_file, 'w', encoding='utf-8') as f:
            # Header
            f.write("=" * 70 + "\n")
            f.write("CALIBRE IMPORT LOG\n")
            f.write("=" * 70 + "\n")
            f.write(f"Date: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
            f.write(f"Script Version: {SCRIPT_VERSION}\n")
            f.write(f"Library: {library_path}\n")
            f.write(f"Total Books Found: {len(azw_files)}\n")
            f.write(f"Timeout per book: 60 seconds\n")
            f.write("\n")
            
            # Failed imports section
            if results['failed'] > 0:
                f.write("-" * 70 + "\n")
                f.write(f"FAILED IMPORTS ({results['failed']})\n")
                f.write("-" * 70 + "\n")
                for book_name, error_msg in results['failed_books']:
                    f.write(f"\n[FAILED] {book_name}\n")
                    f.write(f"  Error: {error_msg if error_msg else 'Unknown error'}\n")
                f.write("\n")
            
            # Timeout section
            if results['timed_out'] > 0:
                f.write("-" * 70 + "\n")
                f.write(f"TIMEOUT IMPORTS ({results['timed_out']})\n")
                f.write("-" * 70 + "\n")
                for book_name in results['timed_out_books']:
                    f.write(f"\n[TIMEOUT] {book_name}\n")
                    f.write(f"  Duration: Exceeded 60 seconds\n")
                    f.write(f"  Note: Book may be stuck in DeDRM processing\n")
                f.write("\n")
            
            # Success section (summary only)
            if results['success'] > 0:
                f.write("-" * 70 + "\n")
                f.write(f"SUCCESSFUL IMPORTS ({results['success']})\n")
                f.write("-" * 70 + "\n")
                f.write(f"Book IDs: {', '.join(results['book_ids'])}\n")
                f.write("\n")
            
            # Summary
            f.write("=" * 70 + "\n")
            f.write("SUMMARY\n")
            f.write("=" * 70 + "\n")
            f.write(f"Total:    {results['total']}\n")
            f.write(f"Success:  {results['success']}\n")
            f.write(f"Failed:   {results['failed']}\n")
            f.write(f"Timeout:  {results['timed_out']}\n")
            f.write("=" * 70 + "\n")
        
        return log_file
        
    except Exception as e:
        print_warn(f"Failed to write log file: {e}")
        return None

def import_all_azw_files(content_dir, library_path, use_duplicates=False, per_book_timeout=120, exclude_asins=None, working_dir=None):
    """
    Import all *.azw files one at a time with individual timeouts
    Optionally exclude ASINs that failed key extraction
    Returns: dict with detailed results
    """
    print_step("Importing ebooks to Calibre...")
    print("--------------------------------------------------")
    print()
    
    # Find all .azw files (excluding failed ASINs)
    print_step("Scanning for .azw files...")
    azw_files = find_all_azw_files(content_dir, exclude_asins=exclude_asins)
    
    if not azw_files:
        print_warn("No .azw files found")
        print()
        return {
            'total': 0,
            'success': 0,
            'failed': 0,
            'timed_out': 0,
            'book_ids': [],
            'failed_books': [],
            'timed_out_books': [],
            'log_file': None
        }
    
    print_ok(f"Found {len(azw_files)} .azw file(s)")
    print()
    
    # Load config to check if raw logging is enabled
    config = load_config()
    enable_raw_logs = config.get('enable_raw_logs', False) if config else False
    
    # Use provided working_dir or determine it
    if working_dir is None:
        script_dir = os.path.dirname(os.path.abspath(__file__))
        user_home = os.path.expanduser("~")
        can_write, _ = check_write_permissions(script_dir)
        working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    # Initialize raw log path if enabled
    raw_log_path = None
    if enable_raw_logs:
        raw_log_path = get_raw_log_path(working_dir, 'import')
        # Cleanup old raw logs (keep 10 most recent)
        logs_dir = os.path.join(working_dir, "Logs", "import_logs")
        cleanup_old_raw_logs(logs_dir, keep_count=10)
    
    # Import each book individually
    results = {
        'total': len(azw_files),
        'success': 0,
        'failed': 0,
        'skipped': 0,
        'timed_out': 0,
        'book_ids': [],
        'failed_books': [],
        'skipped_books': [],
        'timed_out_books': [],
        'log_file': None
    }
    
    print_step(f"Importing books (timeout: {per_book_timeout // 60} minutes per book)...")
    print()
    
    for idx, book_path in enumerate(azw_files, 1):
        book_name = os.path.basename(book_path)
        # Extract ASIN from filename (e.g., B00JBVUJM8_EBOK.azw -> B00JBVUJM8_EBOK)
        asin = os.path.splitext(book_name)[0]
        
        print(f"[{idx}/{len(azw_files)}] {book_name}...")
        
        success, book_id, error_msg, timed_out, already_exists, book_title = import_single_book(
            book_path, 
            library_path, 
            use_duplicates, 
            per_book_timeout,
            raw_log_path=raw_log_path
        )
        
        if success and book_id:
            print_ok(f"  ✓ Imported (ID: {book_id})")
            results['success'] += 1
            results['book_ids'].append(book_id)
        elif timed_out:
            print_error(f"  ⏱ TIMEOUT")
            results['timed_out'] += 1
            results['timed_out_books'].append(book_name)
        elif already_exists:
            print_colored(f"  [SKIPPED] ✗ Book already exists in the database", 'yellow')
            results['skipped'] += 1
            results['skipped_books'].append((asin, book_title if book_title else "Unknown Title"))
        else:
            print_error(f"  ✗ FAILED")
            results['failed'] += 1
            results['failed_books'].append((book_name, error_msg))
    
    print()
    
    # Write log file if there were any failures or timeouts
    if results['failed'] > 0 or results['timed_out'] > 0:
        log_file = write_import_log(library_path, results, azw_files, working_dir)
        results['log_file'] = log_file
    
    # Show raw log location if enabled
    if enable_raw_logs and raw_log_path:
        print_step(f"Raw debug log saved to:")
        print(f"      {raw_log_path}")
        print()
    
    return results

def display_import_results(results):
    """
    Display import results to user
    Handles both old format (subprocess result) and new format (dict)
    """
    print("--------------------------------------------------")
    print_step("Import Results:")
    print()
    
    if not results:
        print_error("Import failed - no results")
        return
    
    # Check if it's the new dict format or old subprocess result
    if isinstance(results, dict):
        # New format from per-book import
        if results['success'] > 0:
            print_ok(f"Successfully imported: {results['success']} ebook(s)")
            if results['book_ids']:
                print(f"      Book IDs: {', '.join(results['book_ids'])}")
        else:
            print_warn("No books were imported")
        
        # Show skipped books (duplicates)
        if results.get('skipped', 0) > 0:
            print()
            print_warn(f"Skipped: {results['skipped']} book(s) already exist in database")
            if results.get('skipped_books'):
                print("      Books that were skipped:")
                for asin, title in results['skipped_books'][:5]:  # Show first 5
                    print(f"        - {asin} - {title}")
                if len(results['skipped_books']) > 5:
                    print(f"        ... and {len(results['skipped_books']) - 5} more")
        
        # Show timeout information
        if results.get('timed_out', 0) > 0:
            print()
            print_error(f"Timed out: {results['timed_out']} book(s)")
            if results.get('timed_out_books'):
                print("      Books that timed out:")
                for book_name in results['timed_out_books'][:5]:  # Show first 5
                    print(f"        - {book_name}")
                if len(results['timed_out_books']) > 5:
                    print(f"        ... and {len(results['timed_out_books']) - 5} more")
        
        # Show failed books
        if results.get('failed', 0) > 0:
            print()
            print_error(f"Failed: {results['failed']} book(s)")
            if results.get('failed_books'):
                print("      Books that failed:")
                for book_name, error in results['failed_books'][:3]:  # Show first 3
                    print(f"        - {book_name}")
                if len(results['failed_books']) > 3:
                    print(f"        ... and {len(results['failed_books']) - 3} more")
        
        # Show log file location if it exists
        if results.get('log_file'):
            print()
            print_step(f"Detailed error log saved to:")
            print(f"      {results['log_file']}")
    
    print()

def is_kindle_running():
    """
    Check if Kindle.exe is currently running on Windows
    Returns: True if Kindle is running, False otherwise
    """
    try:
        # Run tasklist command to check for Kindle.exe
        result = subprocess.run(
            ['tasklist', '/FI', 'IMAGENAME eq Kindle.exe'],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        # Check if Kindle.exe is in the output
        output_lower = result.stdout.lower()
        return 'kindle.exe' in output_lower
        
    except Exception as e:
        # If detection fails, assume Kindle might be running (safer approach)
        print_warn(f"Could not detect Kindle process: {e}")
        return True

def is_calibre_running():
    """
    Check if any Calibre processes are currently running on Windows
    Returns: True if Calibre is running, False otherwise
    """
    try:
        # Run tasklist command to get all running processes
        result = subprocess.run(
            ['tasklist', '/FI', 'IMAGENAME eq calibre*'],
            capture_output=True,
            text=True,
            timeout=5
        )
        
        # Check for common Calibre process names
        calibre_processes = [
            'calibre.exe',
            'calibre-parallel.exe',
            'calibredb.exe',
            'ebook-convert.exe'
        ]
        
        output_lower = result.stdout.lower()
        for process in calibre_processes:
            if process.lower() in output_lower:
                return True
        
        return False
        
    except Exception as e:
        # If detection fails, assume Calibre might be running (safer approach)
        print_warn(f"Could not detect Calibre processes: {e}")
        return True

def verify_calibre_closed_with_loop():
    """
    Verify Calibre is closed with auto-detection loop
    Waits until Calibre closes, no manual intervention required
    Returns: True if Calibre is closed, False if user wants to exit
    """
    if not is_calibre_running():
        print_ok("Calibre confirmed closed - proceeding")
        print()
        return True
    
    # Calibre is still running - wait for it to close
    print_warn("Calibre is still running!")
    print()
    print_step("Waiting for Calibre to close...")
    print("  (Script will automatically continue when Calibre closes)")
    print()
    
    # Auto-detection loop - wait until Calibre closes
    while is_calibre_running():
        time.sleep(1)  # Check every second
    
    print_ok("Calibre has closed - continuing with script...")
    print()
    return True

def warn_close_calibre():
    """
    Warn user to close Calibre before import
    Auto-detects if Calibre is running and skips prompt if not
    Auto-continues once Calibre is detected as closed
    Returns: True if Calibre not running or successfully closed, False if user quits
    """
    # Check if Calibre is running
    if not is_calibre_running():
        print_ok("Calibre not detected - proceeding automatically")
        print()
        return True
    
    print_warn("IMPORTANT: Calibre must be closed for import to work!")
    print()
    print("Please close Calibre now if it's running.")
    print("Importing while Calibre is open may cause:")
    print("  - Database lock errors")
    print("  - Import failures")
    print("  - Data corruption")
    print()
    print_warn("BE AWARE:")
    print("This will attempt to Import ALL BOOKS found ")
    print("in your 'My Kindle Content' location and with any luck they")
    print("should be DeDRM and end up in 'KFX' & 'EPUB' format (If you selected to convert to EPUB)")
    print("-- Good Luck --")
    print()
    
    print_step("Waiting for Calibre to close...")
    print("  (Script will automatically continue when Calibre closes)")
    print()
    
    # Auto-detection loop - wait until Calibre closes
    while is_calibre_running():
        time.sleep(1)  # Check every second
    
    print_ok("Calibre has closed - continuing with script...")
    print()
    return True

# ============================================================================
# KFX TO EPUB CONVERSION FUNCTIONS
# ============================================================================

def query_book_info_from_db(library_path, book_ids):
    """
    Query Calibre database for book titles, authors, and paths
    Returns: dict mapping book_id -> {'title': str, 'author': str, 'path': str}
    """
    import sqlite3
    
    db_path = os.path.join(library_path, "metadata.db")
    book_info = {}
    
    try:
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        
        for book_id in book_ids:
            # Get title and path
            cursor.execute("SELECT title, path FROM books WHERE id = ?", (book_id,))
            result = cursor.fetchone()
            
            if result:
                title = result[0]
                path = result[1]
                
                # Get author
                cursor.execute("""
                    SELECT authors.name 
                    FROM authors 
                    JOIN books_authors_link ON authors.id = books_authors_link.author
                    WHERE books_authors_link.book = ?
                    LIMIT 1
                """, (book_id,))
                author_result = cursor.fetchone()
                author = author_result[0] if author_result else "Unknown Author"
                
                book_info[book_id] = {
                    'title': title,
                    'author': author,
                    'path': path
                }
        
        conn.close()
        return book_info
        
    except Exception as e:
        print_error(f"Failed to query database: {e}")
        return {}

def query_book_paths_from_db(library_path, book_ids):
    """
    Query Calibre metadata.db to get book paths for given book IDs
    Returns: dict mapping book_id -> book_path
    (Legacy function - use query_book_info_from_db for enhanced info)
    """
    book_info = query_book_info_from_db(library_path, book_ids)
    return {book_id: info['path'] for book_id, info in book_info.items()}

def find_source_file_in_directory(book_dir):
    """
    Find the source ebook file in the book directory.
    Searches for Kindle format files: .kfx, .azw, .azw3, .kfx-zip, or .mobi
    
    Args:
        book_dir: Path to the book directory in Calibre library
    
    Returns:
        tuple: (filename, is_kfx_zip) or (None, False) if not found
               - filename: Name of the source file found
               - is_kfx_zip: True if file is .kfx-zip format, False otherwise
    """
    try:
        if not os.path.exists(book_dir):
            return None, False
        
        for filename in os.listdir(book_dir):
            lower_name = filename.lower()
            if lower_name.endswith(('.kfx', '.azw', '.azw3', '.kfx-zip', '.mobi')):
                is_kfx_zip = lower_name.endswith('.kfx-zip')
                return filename, is_kfx_zip
        
        return None, False
        
    except Exception as e:
        print_error(f"Error searching directory {book_dir}: {e}")
        return None, False

def convert_book_to_epub(source_path, epub_path, raw_log_path=None, book_info=None):
    """
    Convert Imported eBook to EPUB using ebook-convert
    Timeout: 3 minutes per book
    Returns: (success: bool, error_message: str, error_type: dict)
    error_type dict contains: {'is_kfx_error': bool, 'is_timeout': bool, 'is_drm': bool, 'is_pdf': bool}
    """
    try:
        cmd = [
            'ebook-convert',
            source_path,
            epub_path,
            '--input-profile=default',
            '--output-profile=tablet',
            '--no-svg-cover',
            '--epub-version=3'
        ]
        
        # Start subprocess with Popen for non-blocking execution
        process = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding='utf-8',
            errors='replace'
        )
        
        # Timer display thread
        timer_stopped = threading.Event()
        timeout_seconds = 180  # 3 minutes
        
        timer_thread = threading.Thread(
            target=display_progress_timer,
            args=("Converting to EPUB", timeout_seconds, timer_stopped),
            daemon=True
        )
        timer_thread.start()
        
        # Wait for process with timeout
        try:
            stdout, stderr = process.communicate(timeout=timeout_seconds)
            result_returncode = process.returncode
            # Stop timer and clear the line on success
            timer_stopped.set()
            timer_thread.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
        except subprocess.TimeoutExpired:
            # Stop timer and clear the line on timeout
            timer_stopped.set()
            timer_thread.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
            process.kill()
            stdout, stderr = process.communicate()
            result_returncode = -1
        
        # Create a result-like object for compatibility
        class Result:
            def __init__(self, returncode, stdout, stderr):
                self.returncode = returncode
                self.stdout = stdout
                self.stderr = stderr
        
        result = Result(result_returncode, stdout, stderr)
        
        # Log to raw log if enabled
        if raw_log_path:
            append_to_raw_log(
                raw_log_path,
                cmd,
                result.stdout,
                result.stderr,
                result.returncode,
                book_info=book_info
            )
        
        if result.returncode == 0 and os.path.exists(epub_path):
            error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': False}
            return True, "", error_type
        else:
            # Combine stdout and stderr for complete error detection
            stdout_text = result.stdout if result.stdout else ""
            stderr_text = result.stderr if result.stderr else ""
            combined_output = stdout_text + "\n" + stderr_text
            error_msg = stderr_text if stderr_text else "Conversion failed"
            
            # Detect various error types
            is_kfx_error = False
            is_drm_error = False
            is_pdf_content = False
            
            if combined_output:
                # Detect PDF content
                pdf_indicators = [
                    "contains PDF content",
                    "Convert to PDF to extract it",
                    "PDF content"
                ]
                is_pdf_content = any(indicator.lower() in combined_output.lower() for indicator in pdf_indicators)
                
                # Detect KFX-specific errors
                kfx_indicators = [
                    "Amazon KFX book",
                    "KFX book",
                    "cannot be processed",
                    "KFXError"
                ]
                is_kfx_error = any(indicator.lower() in combined_output.lower() for indicator in kfx_indicators)
                
                # Detect DRM-specific errors
                drm_indicators = [
                    "DRMError",
                    "has DRM",
                    "KFXDRMError"
                ]
                is_drm_error = any(indicator.lower() in combined_output.lower() for indicator in drm_indicators)
                
                # Provide user-friendly error messages
                if is_pdf_content:
                    error_msg = (
                        "Book contains PDF content - cannot convert to EPUB\n"
                        "          This book uses PDF format which cannot be converted to EPUB\n"
                        "          The book will remain in its original format"
                    )
                elif is_kfx_error:
                    error_msg = (
                        "Amazon KFX book - requires KFX Input plugin to be installed\n"
                        "          See: https://www.mobileread.com/forums/showthread.php?t=283371"
                    )
                elif is_drm_error:
                    error_msg = (
                        "Book has DRM and cannot be converted\n"
                        "          Possible causes:\n"
                        "            - DeDRM plugin not installed\n"
                        "            - Key extraction failed (no valid decryption key found)\n"
                        "          Install DeDRM plugin: https://github.com/Satsuoni/DeDRM_tools"
                    )
                    # Treat DRM errors as KFX errors for tracking purposes (both are DRM-related)
                    is_kfx_error = True
            
            error_type = {'is_kfx_error': is_kfx_error, 'is_timeout': False, 'is_drm': is_drm_error, 'is_pdf': is_pdf_content}
            return False, error_msg, error_type
            
    except subprocess.TimeoutExpired:
        # Log timeout to raw log if enabled
        if raw_log_path:
            append_to_raw_log(
                raw_log_path,
                cmd, # pyright: ignore[reportPossiblyUnboundVariable]
                '',
                '',
                -1,  # Timeout exit code
                book_info=book_info
            )
        error_type = {'is_kfx_error': False, 'is_timeout': True, 'is_drm': False, 'is_pdf': False}
        return False, "Conversion timeout (3 minutes) - book may be too large or complex", error_type
    except Exception as e:
        error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': False}
        return False, str(e), error_type

def convert_azw3_via_mobi(source_path, epub_path, working_dir=None):
    """
    Convert AZW3 to EPUB via temporary MOBI intermediate format
    Two-step conversion: AZW3 → MOBI → EPUB
    
    This produces better results than direct AZW3 → EPUB conversion
    because MOBI acts as a better intermediate format for preserving
    layout and formatting.
    
    Temp MOBI file is created in temp_extraction folder for automatic
    cleanup by cleanup_temp_extraction() on next script run.
    
    Args:
        source_path: Path to source AZW3 file
        epub_path: Path to output EPUB file
        working_dir: Directory to use for temp files (respects fallback paths)
    
    Returns:
        tuple: (success: bool, error_message: str, is_kfx_error: bool)
    """
    if working_dir is None:
        # Determine working_dir if not provided
        script_dir = os.path.dirname(os.path.abspath(__file__))
        user_home = os.path.expanduser("~")
        can_write, _ = check_write_permissions(script_dir)
        working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
    
    temp_extraction_dir = os.path.join(working_dir, "temp_extraction")
    
    # Fixed filename - safe because conversions happen sequentially
    temp_mobi_path = os.path.join(temp_extraction_dir, "temp_conversion.mobi")
    
    try:
        # Ensure temp directory exists
        os.makedirs(temp_extraction_dir, exist_ok=True)
        
        # Step 1: Convert AZW3 to MOBI
        cmd_mobi = [
            'ebook-convert',
            source_path,
            temp_mobi_path,
            '--input-profile=default',
            '--output-profile=tablet'
        ]
        
        # Start Step 1 subprocess
        process_mobi = subprocess.Popen(
            cmd_mobi,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding='utf-8',
            errors='replace'
        )
        
        # Timer display thread for Step 1
        timer_stopped_step1 = threading.Event()
        timeout_seconds_step1 = 300
        
        timer_thread_step1 = threading.Thread(
            target=display_progress_timer,
            args=("Step 1/2 (AZW3→MOBI)", timeout_seconds_step1, timer_stopped_step1),
            daemon=True
        )
        timer_thread_step1.start()
        
        # Wait for Step 1 with timeout
        try:
            stdout_mobi, stderr_mobi = process_mobi.communicate(timeout=timeout_seconds_step1)
            result_mobi_returncode = process_mobi.returncode
            # Stop timer and clear the line
            timer_stopped_step1.set()
            timer_thread_step1.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
        except subprocess.TimeoutExpired:
            # Stop timer and clear the line
            timer_stopped_step1.set()
            timer_thread_step1.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
            process_mobi.kill()
            stdout_mobi, stderr_mobi = process_mobi.communicate()
            return False, "Step 1 timeout (5 minutes)", False
        
        # Create result-like object for Step 1
        class Result:
            def __init__(self, returncode, stdout, stderr):
                self.returncode = returncode
                self.stdout = stdout
                self.stderr = stderr
        
        result_mobi = Result(result_mobi_returncode, stdout_mobi, stderr_mobi)
        
        if result_mobi.returncode != 0 or not os.path.exists(temp_mobi_path):
            error_msg = result_mobi.stderr if result_mobi.stderr else "AZW3 to MOBI conversion failed"
            return False, f"Step 1 failed: {error_msg}", False
        
        # Step 2: Convert MOBI to EPUB
        cmd_epub = [
            'ebook-convert',
            temp_mobi_path,
            epub_path,
            '--input-profile=default',
            '--output-profile=tablet',
            '--no-svg-cover',
            '--epub-version=3'
        ]
        
        # Start Step 2 subprocess
        process_epub = subprocess.Popen(
            cmd_epub,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding='utf-8',
            errors='replace'
        )
        
        # Timer display thread for Step 2
        timer_stopped_step2 = threading.Event()
        timeout_seconds_step2 = 300
        
        timer_thread_step2 = threading.Thread(
            target=display_progress_timer,
            args=("Step 2/2 (MOBI→EPUB)", timeout_seconds_step2, timer_stopped_step2),
            daemon=True
        )
        timer_thread_step2.start()
        
        # Wait for Step 2 with timeout
        try:
            stdout_epub, stderr_epub = process_epub.communicate(timeout=timeout_seconds_step2)
            result_epub_returncode = process_epub.returncode
            # Stop timer and clear the line
            timer_stopped_step2.set()
            timer_thread_step2.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
        except subprocess.TimeoutExpired:
            # Stop timer and clear the line
            timer_stopped_step2.set()
            timer_thread_step2.join(timeout=1)
            print("\r" + " " * 80 + "\r", end='', flush=True)
            process_epub.kill()
            stdout_epub, stderr_epub = process_epub.communicate()
            return False, "Step 2 timeout (5 minutes)", False
        
        # Create result-like object for Step 2
        result_epub = Result(result_epub_returncode, stdout_epub, stderr_epub)
        
        if result_epub.returncode == 0 and os.path.exists(epub_path):
            return True, "", False
        else:
            error_msg = result_epub.stderr if result_epub.stderr else "MOBI to EPUB conversion failed"
            return False, f"Step 2 failed: {error_msg}", False
            
    except subprocess.TimeoutExpired:
        return False, "Conversion timeout (5 minutes per step)", False
    except Exception as e:
        return False, str(e), False
    finally:
        # Always try to cleanup temp MOBI file
        if os.path.exists(temp_mobi_path):
            try:
                os.remove(temp_mobi_path)
            except Exception:
                pass  # If cleanup fails, cleanup_temp_extraction() will handle it on next run

def add_epub_format_to_calibre(book_id, epub_path, library_path):
    """
    Add EPUB format to existing Calibre book record using calibredb add_format
    Returns: (success: bool, error_message: str)
    """
    try:
        cmd = [
            'calibredb', 'add_format',
            str(book_id),
            epub_path,
            f'--library-path={library_path}'
        ]
        
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=60)
        
        if result.returncode == 0:
            return True, ""
        else:
            error_msg = result.stderr if result.stderr else "Failed to add format"
            return False, error_msg
            
    except subprocess.TimeoutExpired:
        return False, "Add format timeout"
    except Exception as e:
        return False, str(e)

def prompt_source_file_management():
    """
    Ask user what to do with source KFX files after successful EPUB conversion
    Returns: 'keep_both' | 'delete_kfx' | 'delete_kfx_zip_only'
    """
    print_step("Source File Management")
    print("--------------------------------------------------")
    print()
    print("After successful EPUB conversion, what should happen to the source KFX files?")
    print()
    print("Options:")
    print("  [K] Keep Both - Preserve both KFX and EPUB formats (recommended)")
    print("  [D] Delete KFX - Remove KFX format after successful EPUB conversion")
    print("  [S] Smart Cleanup - Delete only .kfx-zip files, keep regular .kfx files")
    print()
    
    while True:
        choice = input("Your choice (K/D/S): ").strip().upper()
        if choice == 'K':
            print()
            print_ok("Will keep both KFX and EPUB formats")
            print()
            return 'keep_both'
        elif choice == 'D':
            print()
            print_warn("Will delete KFX format after successful EPUB conversion")
            print()
            return 'delete_kfx'
        elif choice == 'S':
            print()
            print_ok("Will delete only .kfx-zip files, keeping regular .kfx files")
            print()
            return 'delete_kfx_zip_only'
        else:
            print_error("Invalid choice. Please enter K, D, or S.")

def remove_format_from_calibre(book_id, format_name, library_path):
    """
    Remove a specific format from Calibre book record using calibredb
    Returns: (success: bool, error_message: str)
    """
    try:
        cmd = [
            'calibredb', 'remove_format',
            str(book_id),
            format_name.upper(),
            f'--library-path={library_path}'
        ]
        
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=60)
        
        if result.returncode == 0:
            return True, ""
        else:
            error_msg = result.stderr if result.stderr else "Failed to remove format"
            return False, error_msg
            
    except subprocess.TimeoutExpired:
        return False, "Remove format timeout"
    except Exception as e:
        return False, str(e)

def write_conversion_log(library_path, stats, book_info, working_dir):
    """
    Write detailed conversion log to file
    Uses working_dir which respects fallback paths
    Returns: log file path
    """
    # Create logs directory with conversion subfolder
    logs_dir = os.path.join(working_dir, "Logs", "conversion_logs")
    os.makedirs(logs_dir, exist_ok=True)
    
    # Create timestamped log file
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    log_file = os.path.join(logs_dir, f"calibre_conversion_{timestamp}.log")
    
    try:
        with open(log_file, 'w', encoding='utf-8') as f:
            # Header
            f.write("=" * 70 + "\n")
            f.write("CALIBRE KFX TO EPUB CONVERSION LOG\n")
            f.write("=" * 70 + "\n")
            f.write(f"Date: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
            f.write(f"Library: {library_path}\n")
            f.write(f"Total Books Processed: {stats['total']}\n")
            f.write(f"Timeout per book: 300 seconds (5 minutes)\n")
            f.write("\n")
            
            # Failed conversions section
            if stats.get('failed_conversions'):
                f.write("-" * 70 + "\n")
                f.write(f"FAILED CONVERSIONS ({len(stats['failed_conversions'])})\n")
                f.write("-" * 70 + "\n")
                for book_id, title, author, error_msg in stats['failed_conversions']:
                    f.write(f"\n[FAILED] {title}\n")
                    f.write(f"  Author: {author}\n")
                    f.write(f"  Book ID: {book_id}\n")
                    f.write(f"  Error: {error_msg}\n")
                f.write("\n")
            
            # Failed merges section
            if stats.get('failed_merges'):
                f.write("-" * 70 + "\n")
                f.write(f"FAILED MERGES TO CALIBRE ({len(stats['failed_merges'])})\n")
                f.write("-" * 70 + "\n")
                for book_id, title, author, error_msg in stats['failed_merges']:
                    f.write(f"\n[FAILED MERGE] {title}\n")
                    f.write(f"  Author: {author}\n")
                    f.write(f"  Book ID: {book_id}\n")
                    f.write(f"  Error: {error_msg}\n")
                f.write("\n")
            
            # Skipped books section
            if stats.get('skipped_books'):
                f.write("-" * 70 + "\n")
                f.write(f"SKIPPED BOOKS ({len(stats['skipped_books'])})\n")
                f.write("-" * 70 + "\n")
                for book_id, title, author, reason in stats['skipped_books']:
                    f.write(f"\n[SKIPPED] {title}\n")
                    f.write(f"  Author: {author}\n")
                    f.write(f"  Book ID: {book_id}\n")
                    f.write(f"  Reason: {reason}\n")
                f.write("\n")
            
            # Success section (summary only)
            if stats['converted'] > 0:
                f.write("-" * 70 + "\n")
                f.write(f"SUCCESSFUL CONVERSIONS ({stats['converted']})\n")
                f.write("-" * 70 + "\n")
                f.write(f"Books successfully converted and merged to EPUB format\n")
                f.write("\n")
            
            # Summary
            f.write("=" * 70 + "\n")
            f.write("SUMMARY\n")
            f.write("=" * 70 + "\n")
            f.write(f"Total:              {stats['total']}\n")
            f.write(f"Converted:          {stats['converted']}\n")
            f.write(f"Merged to Calibre:  {stats['merged']}\n")
            f.write(f"Failed:             {stats['failed']}\n")
            f.write(f"Skipped:            {stats.get('skipped_kfx_zip', 0)}\n")
            if stats.get('source_files_deleted', 0) > 0:
                f.write(f"Source Files Deleted: {stats['source_files_deleted']}\n")
            f.write("=" * 70 + "\n")
        
        return log_file
        
    except Exception as e:
        print_warn(f"Failed to write conversion log file: {e}")
        return None

def convert_book_alternate_method(source_path, epub_path, library_path, raw_log_path=None, book_info=None):
    """
    Alternate conversion method using calibre-debug -r "KFX Input"
    This method is more robust for complex/large books that timeout with ebook-convert
    
    Args:
        source_path: Path to source KFX/AZW file
        epub_path: Path to output EPUB file
        library_path: Path to Calibre library (for context)
        raw_log_path: Optional path to raw debug log file
        book_info: Optional book identifier for logging
    
    Returns:
        tuple: (success: bool, error_message: str, error_type: dict)
    """
    try:
        # First, check if source is PDF (KFX Input can't handle PDFs)
        if source_path.lower().endswith('.pdf'):
            # Use standard ebook-convert for PDFs
            cmd = [
                'ebook-convert',
                source_path,
                epub_path,
                '--input-profile=default',
                '--output-profile=tablet'
            ]
        else:
            # Use calibre-debug with KFX Input plugin for KFX/AZW files
            cmd = [
                'calibre-debug',
                '-r',
                'KFX Input',
                '--',
                source_path,
                epub_path
            ]
        
        # Run with 5-minute timeout, capturing output silently
        process = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding='utf-8',
            errors='replace',
            bufsize=1
        )
        
        stdout_lines = []
        stderr_lines = []
        
        # Read output silently in background
        import sys
        import threading
        
        pdf_detected = threading.Event()  # Flag to detect PDF content early
        
        # Silent reader threads (no console output)
        def read_silently(pipe, lines_list):
            try:
                for line in iter(pipe.readline, ''):
                    if line:
                        lines_list.append(line)
                        # Check for PDF content warning - ABORT IMMEDIATELY
                        if "contains PDF content" in line or "Use the --pdf option" in line:
                            pdf_detected.set()
                            # Kill the process immediately to stop wasting time on EPUB conversion
                            try:
                                process.kill()
                            except:
                                pass
                            return  # Exit thread
            except Exception:
                pass
        
        # Start silent reader threads
        stdout_thread = threading.Thread(target=read_silently, args=(process.stdout, stdout_lines))
        stderr_thread = threading.Thread(target=read_silently, args=(process.stderr, stderr_lines))
        
        stdout_thread.daemon = True
        stderr_thread.daemon = True
        
        stdout_thread.start()
        stderr_thread.start()
        
        # Timer display thread for alternate method
        timer_stopped = threading.Event()
        timeout_seconds = 300  # 5 minutes
        
        def display_alternate_timer():
            start_time = time.time()
            while not timer_stopped.is_set() and not pdf_detected.is_set():
                elapsed = int(time.time() - start_time)
                minutes, seconds = divmod(elapsed, 60)
                timeout_mins, timeout_secs = divmod(timeout_seconds, 60)
                print(f"\r  Converting (Alternate).. [ {minutes:02d}:{seconds:02d} / {timeout_mins:02d}:{timeout_secs:02d} ]", 
                      end='', flush=True)
                time.sleep(1)
        
        timer_thread = threading.Thread(target=display_alternate_timer, daemon=True)
        timer_thread.start()
        
        # Wait for process with timeout OR until PDF is detected
        start_time = time.time()
        timed_out = False
        
        try:
            # Poll for process completion or PDF detection
            while process.poll() is None:
                if pdf_detected.is_set():
                    # PDF detected - process should already be killed by thread
                    break
                
                # Check if we've exceeded the timeout
                if time.time() - start_time > timeout_seconds:
                    timed_out = True
                    process.kill()
                    break
                
                time.sleep(0.1)
            
            # If we timed out, return timeout error
            if timed_out:
                # Stop timer and clear the line
                timer_stopped.set()
                timer_thread.join(timeout=1)
                print("\r" + " " * 80 + "\r", end='', flush=True)
                # Log timeout to raw log if enabled
                if raw_log_path:
                    append_to_raw_log(
                        raw_log_path,
                        cmd,
                        ''.join(stdout_lines),
                        ''.join(stderr_lines),
                        -1,  # Timeout exit code
                        book_info=book_info
                    )
                error_type = {'is_kfx_error': False, 'is_timeout': True, 'is_drm': False, 'is_pdf': False}
                return False, "Alternate conversion timeout (5 minutes)", error_type
        except:
            pass
        
        # Stop timer and clear the line
        timer_stopped.set()
        timer_thread.join(timeout=1)
        print("\r" + " " * 80 + "\r", end='', flush=True)
        
        # Wait for threads to finish
        stdout_thread.join(timeout=2)
        stderr_thread.join(timeout=2)
        
        # Check if PDF was detected during processing
        if pdf_detected.is_set():
            # Inform user about PDF detection
            print()
            print_warn("  ⚠   PDF CONTENT DETECTED - Cannot convert to EPUB")
            print_colored("  → Switching to PDF extraction mode...", 'cyan')
            print()
            
            # Try to extract as PDF instead
            pdf_path = epub_path.replace('.epub', '.pdf')
            
            cmd_pdf = [
                'calibre-debug',
                '-r',
                'KFX Input',
                '--',
                '-p',
                source_path,
                pdf_path
            ]
            
            try:
                # Run PDF extraction silently with timer
                process_pdf = subprocess.Popen(
                    cmd_pdf,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE,
                    text=True,
                    encoding='utf-8',
                    errors='replace',
                    bufsize=1
                )
                
                stdout_pdf_lines = []
                stderr_pdf_lines = []
                
                # Silent PDF reader threads
                def read_pdf_silently(pipe, lines_list):
                    try:
                        for line in iter(pipe.readline, ''):
                            if line:
                                lines_list.append(line)
                    except Exception:
                        pass
                
                stdout_pdf_thread = threading.Thread(target=read_pdf_silently, args=(process_pdf.stdout, stdout_pdf_lines))
                stderr_pdf_thread = threading.Thread(target=read_pdf_silently, args=(process_pdf.stderr, stderr_pdf_lines))
                
                stdout_pdf_thread.daemon = True
                stderr_pdf_thread.daemon = True
                
                stdout_pdf_thread.start()
                stderr_pdf_thread.start()
                
                # PDF timer display
                pdf_timer_stopped = threading.Event()
                
                def display_pdf_timer():
                    start_time = time.time()
                    while not pdf_timer_stopped.is_set():
                        elapsed = int(time.time() - start_time)
                        minutes, seconds = divmod(elapsed, 60)
                        timeout_mins, timeout_secs = divmod(300, 60)
                        print(f"\r  Extracting PDF.. [ {minutes:02d}:{seconds:02d} / {timeout_mins:02d}:{timeout_secs:02d} ]", 
                              end='', flush=True)
                        time.sleep(1)
                
                pdf_timer_thread = threading.Thread(target=display_pdf_timer, daemon=True)
                pdf_timer_thread.start()
                
                # Wait for process with timeout
                try:
                    process_pdf.wait(timeout=300)
                except subprocess.TimeoutExpired:
                    process_pdf.kill()
                    pdf_timer_stopped.set()
                    pdf_timer_thread.join(timeout=1)
                    print("\r" + " " * 80 + "\r", end='', flush=True)
                    error_type = {'is_kfx_error': False, 'is_timeout': True, 'is_drm': False, 'is_pdf': True}
                    return False, "PDF extraction timeout (5 minutes)", error_type
                
                # Stop timer and clear line
                pdf_timer_stopped.set()
                pdf_timer_thread.join(timeout=1)
                print("\r" + " " * 80 + "\r", end='', flush=True)
                
                # Wait for threads
                stdout_pdf_thread.join(timeout=2)
                stderr_pdf_thread.join(timeout=2)
                
                # Combine output for logging
                stdout_pdf_text = ''.join(stdout_pdf_lines)
                stderr_pdf_text = ''.join(stderr_pdf_lines)
                
                # Log PDF extraction to raw log if enabled
                if raw_log_path:
                    append_to_raw_log(
                        raw_log_path,
                        cmd_pdf,
                        stdout_pdf_text,
                        stderr_pdf_text,
                        process_pdf.returncode,
                        book_info=f"{book_info} (PDF extraction)"
                    )
                
                if process_pdf.returncode == 0 and os.path.exists(pdf_path):
                    # Return success with PDF indicator - calling code will handle PDF format merge
                    error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': True}
                    return True, "", error_type  # Success, but it's a PDF
                else:
                    error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': True}
                    error_msg = (
                        "Book contains PDF content - PDF extraction failed\n"
                        "          Attempted to extract as PDF but conversion failed\n"
                        "          The book will remain in its original format"
                    )
                    return False, error_msg, error_type
                    
            except subprocess.TimeoutExpired:
                error_type = {'is_kfx_error': False, 'is_timeout': True, 'is_drm': False, 'is_pdf': True}
                return False, "PDF extraction timeout (5 minutes)", error_type
            except Exception as e:
                error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': True}
                return False, f"PDF extraction failed: {e}", error_type
        
        # Combine output
        stdout_text = ''.join(stdout_lines)
        stderr_text = ''.join(stderr_lines)
        combined_output = stdout_text + "\n" + stderr_text
        
        # Log to raw log if enabled
        if raw_log_path:
            append_to_raw_log(
                raw_log_path,
                cmd,
                stdout_text,
                stderr_text,
                process.returncode,
                book_info=book_info
            )
        
        if process.returncode == 0 and os.path.exists(epub_path):
            error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': False}
            return True, "", error_type
        else:
            error_msg = stderr_text if stderr_text else "Alternate conversion failed"
            
            # Detect error types
            is_kfx_error = False
            is_drm_error = False
            is_pdf_content = False
            
            if combined_output:
                # Detect PDF content
                pdf_indicators = ["contains PDF content", "Convert to PDF to extract it", "PDF content"]
                is_pdf_content = any(indicator.lower() in combined_output.lower() for indicator in pdf_indicators)
                
                # Detect KFX-specific errors
                kfx_indicators = ["KFX", "cannot be processed", "KFXError"]
                is_kfx_error = any(indicator.lower() in combined_output.lower() for indicator in kfx_indicators)
                
                # Detect DRM-specific errors
                drm_indicators = ["DRMError", "has DRM", "KFXDRMError"]
                is_drm_error = any(indicator.lower() in combined_output.lower() for indicator in drm_indicators)
            
            error_type = {'is_kfx_error': is_kfx_error, 'is_timeout': False, 'is_drm': is_drm_error, 'is_pdf': is_pdf_content}
            return False, error_msg, error_type
            
    except Exception as e:
        error_type = {'is_kfx_error': False, 'is_timeout': False, 'is_drm': False, 'is_pdf': False}
        return False, str(e), error_type

def process_book_conversions(library_path, book_ids, calibre_config=None, working_dir=None):
    """
    Main orchestrator for Phase 4: KFX to EPUB conversion
    Returns: dict with conversion statistics
    """
    # Load config to check clear screen setting
    config = load_config()
    if config and config.get('clear_screen_between_phases', True):
        os.system('cls')
    
    display_phase_banner(4, "Imported eBook to EPUB Conversion")
    
    stats = {
        'total': len(book_ids),
        'converted': 0,
        'merged': 0,
        'failed': 0,
        'errors': [],
        'source_files_deleted': 0,
        'failed_conversions': [],
        'failed_merges': [],
        'skipped_books': []
    }
    
    if not book_ids:
        print_warn("No books to convert")
        return stats
    
    # Use passed config or load from unified config
    if calibre_config is None:
        unified_config = load_config()
        calibre_config = unified_config.get('calibre_import', {}) if unified_config else {}
    
    # Get settings from config
    kfx_zip_mode = calibre_config.get('kfx_zip_mode', 'convert_all')
    skip_kfx_zip = (kfx_zip_mode == 'skip_kfx_zip')
    source_management = calibre_config.get('source_file_management', 'keep_both')
    
    print_step(f"Converting {len(book_ids)} book(s) to EPUB format...")
    print()
    print_warn("NOTE: Each book has a 3-minute timeout for conversion")
    print("      Large or complex books may timeout and fail")
    print()
    
    # Query database for book info (titles, authors, paths)
    print_step("Querying Calibre database for book information...")
    book_info = query_book_info_from_db(library_path, book_ids)
    
    if not book_info:
        print_error("Failed to retrieve book information from database")
        return stats
    
    print_ok(f"Retrieved information for {len(book_info)} book(s)")
    print()
    
    # Add tracking for skipped, DRM-protected, and timeout files
    stats['skipped_kfx_zip'] = 0
    stats['failed_drm_protected'] = 0
    stats['failed_timeout'] = 0
    stats['timeout_books'] = []  # List of tuples: (book_id, title, author)
    
    # Process each book
    for idx, (book_id, info) in enumerate(book_info.items(), 1):
        title = info['title']
        author = info['author']
        book_path = info['path']
        
        print_step(f"Processing book {idx}/{len(book_info)}: '{title}' by {author}")
        
        # Construct full path to book directory
        book_dir = os.path.join(library_path, book_path)
        
        # Find source file (returns tuple: filename, is_kfx_zip)
        source_filename, is_kfx_zip = find_source_file_in_directory(book_dir)
        
        if not source_filename:
            error_msg = f"No source file (KFX/AZW/AZW3/KFX-ZIP/MOBI) found in {book_path}"
            print_error(error_msg)
            stats['failed'] += 1
            stats['errors'].append(f"Book {book_id}: {error_msg}")
            stats['failed_conversions'].append((book_id, title, author, error_msg))
            print()
            continue
        
        # Check if we should skip .kfx-zip files
        if is_kfx_zip and skip_kfx_zip:
            print(f"  Source: {source_filename}")
            print(f"  Skipping .kfx-zip file (DRM-protected)")
            stats['skipped_kfx_zip'] += 1
            stats['skipped_books'].append((book_id, title, author, "KFX-ZIP file (DRM-protected)"))
            print()
            continue
        
        source_path = os.path.join(book_dir, source_filename)
        epub_filename = os.path.splitext(source_filename)[0] + '.epub'
        epub_path = os.path.join(book_dir, epub_filename)
        
        print(f"  Source: {source_filename}")
        print(f"  Target: {epub_filename}")
        
        # Detect format for conversion routing
        is_azw3 = source_filename.lower().endswith('.azw3')
        is_mobi = source_filename.lower().endswith('.mobi')
        is_azw = source_filename.lower().endswith('.azw')
        
        # Use passed working_dir parameter or determine it if not provided
        if working_dir is None:
            script_dir = os.path.dirname(os.path.abspath(__file__))
            user_home = os.path.expanduser("~")
            can_write, _ = check_write_permissions(script_dir)
            working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
        
        # Initialize raw log path if enabled
        enable_raw_logs = config.get('enable_raw_logs', False) if config else False
        raw_log_path = None
        if enable_raw_logs:
            raw_log_path = get_raw_log_path(working_dir, 'conversion')
            # Cleanup old raw logs (keep 10 most recent)
            logs_dir = os.path.join(working_dir, "Logs", "conversion_logs")
            cleanup_old_raw_logs(logs_dir, keep_count=10)
        
        # Convert to EPUB (route based on format)
        if is_azw3:
            # AZW3 uses MOBI intermediate for better conversion
            print("  Converting to EPUB (via MOBI intermediate)...")
            success, error, is_kfx_error = convert_azw3_via_mobi(source_path, epub_path, working_dir=working_dir)
            error_type = {'is_kfx_error': is_kfx_error, 'is_timeout': False, 'is_drm': False}
        elif is_mobi or is_azw:
            # MOBI and AZW convert directly to EPUB
            print("  Converting to EPUB...")
            book_info_str = f"{book_id} - {title} by {author}"
            success, error, error_type = convert_book_to_epub(source_path, epub_path, raw_log_path=raw_log_path, book_info=book_info_str)
        else:
            # KFX and KFX-ZIP use standard conversion
            print("  Converting to EPUB...")
            book_info_str = f"{book_id} - {title} by {author}"
            success, error, error_type = convert_book_to_epub(source_path, epub_path, raw_log_path=raw_log_path, book_info=book_info_str)
        
        if not success:
            # Add timeout indicator to error message if it was a timeout
            if error_type.get('is_timeout'):
                print_error(f"  Conversion failed - Timeout")
            else:
                print_error(f"  Conversion failed")
            print_error(f"  {error}")
            stats['failed'] += 1
            # Track if it's a KFX-specific error or DRM-protected file
            if error_type.get('is_kfx_error') or error_type.get('is_drm') or is_kfx_zip:
                stats['failed_drm_protected'] += 1
            # Track timeout failures separately
            if error_type.get('is_timeout'):
                stats['failed_timeout'] += 1
                stats['timeout_books'].append((book_id, title, author))
            stats['errors'].append(f"Book {book_id}: Conversion failed - {error}")
            stats['failed_conversions'].append((book_id, title, author, error))
            print()
            continue
        
        print_ok("  Conversion successful")
        stats['converted'] += 1
        
        # Merge EPUB format into Calibre
        print("  Merging EPUB format to Calibre...")
        success, error = add_epub_format_to_calibre(book_id, epub_path, library_path)
        
        if not success:
            print_error(f"  {error}")
            stats['errors'].append(f"Book {book_id}: Failed to merge format - {error}")
            stats['failed_merges'].append((book_id, title, author, error))
        else:
            print_ok("  EPUB format merged successfully")
            stats['merged'] += 1
            
            # Handle source file management based on user choice
            # Only delete KFX and KFX-ZIP formats (preserve MOBI/AZW/AZW3)
            if source_management == 'delete_source':
                # Determine the actual source format
                source_ext = os.path.splitext(source_filename)[1].upper().replace('.', '')
                
                # Only delete KFX and KFX-ZIP formats
                if source_ext in ['KFX', 'KFX-ZIP']:
                    print(f"  Removing {source_ext} format from Calibre...")
                    success, error = remove_format_from_calibre(book_id, source_ext, library_path)
                    if success:
                        print_ok(f"  {source_ext} format removed")
                        stats['source_files_deleted'] += 1
                    else:
                        print_warn(f"  Failed to remove {source_ext} format: {error}")
                else:
                    print(f"  Keeping {source_ext} format (only KFX/KFX-ZIP are deleted)")
            
            elif source_management == 'delete_kfx_zip_only' and is_kfx_zip:
                # Delete only .kfx-zip files
                print("  Removing KFX-ZIP format from Calibre...")
                success, error = remove_format_from_calibre(book_id, 'KFX-ZIP', library_path)
                if success:
                    print_ok("  KFX-ZIP format removed")
                    stats['source_files_deleted'] += 1
                else:
                    print_warn(f"  Failed to remove KFX-ZIP format: {error}")
        
        print()
    
    # Display summary
    print("--------------------------------------------------")
    print_step("Conversion Summary:")
    print()
    print_ok(f"Total books processed: {stats['total']}")
    print_ok(f"Successfully converted: {stats['converted']}")
    print_ok(f"Successfully merged to Calibre: {stats['merged']}")
    
    if stats['failed'] > 0:
        print_error(f"Failed conversions: {stats['failed']}")
        if stats['failed_drm_protected'] > 0:
            print_error(f"  Failed (likely DRM-protected): {stats['failed_drm_protected']}")
    
    if stats['skipped_kfx_zip'] > 0:
        print_warn(f"Skipped (.kfx-zip files): {stats['skipped_kfx_zip']}")
    
    if stats['source_files_deleted'] > 0:
        print_ok(f"Source files removed: {stats['source_files_deleted']}")
    
    # Write log file if there were any failures or skipped books
    if stats['failed'] > 0 or stats.get('skipped_kfx_zip', 0) > 0 or stats.get('failed_merges'):
        # Use passed working_dir parameter or determine it if not provided
        if working_dir is None:
            user_home = os.path.expanduser("~")
            script_dir = os.path.dirname(os.path.abspath(__file__))
            can_write, _ = check_write_permissions(script_dir)
            working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
        log_file = write_conversion_log(library_path, stats, book_info, working_dir)
        if log_file:
            print()
            print_step(f"Detailed conversion log saved to:")
            print(f"      {log_file}")
    
    # Display Phase 4 summary
    summary_points = [
        f"Processed {stats['total']} book(s) for EPUB conversion",
        f"Successfully converted: {stats['converted']} book(s)",
        f"Successfully merged to Calibre: {stats['merged']} book(s)"
    ]
    
    if stats['failed'] > 0:
        summary_points.append(f"Failed conversions: {stats['failed']} book(s)")
    
    if stats['skipped_kfx_zip'] > 0:
        summary_points.append(f"Skipped .kfx-zip files: {stats['skipped_kfx_zip']}")
    
    if stats['source_files_deleted'] > 0:
        summary_points.append(f"Source files removed: {stats['source_files_deleted']}")
    
    display_phase_summary(4, "KFX to EPUB Conversion", summary_points, pause_seconds=5)
    
    # Cleanup temp_extraction folder after all conversions complete
    cleanup_temp_extraction(silent=True, working_dir=working_dir)
    
    return stats

def process_timeout_retries(library_path, timeout_books, book_info, calibre_config=None):
    """
    Phase 4B: Retry timed-out conversions using alternate method
    
    Args:
        library_path: Path to Calibre library
        timeout_books: List of tuples (book_id, title, author) that timed out
        book_info: Dict mapping book_id -> {'title': str, 'author': str, 'path': str}
        calibre_config: Calibre configuration dict
    
    Returns:
        dict with retry statistics
    """
    # Load config to check clear screen setting
    config = load_config()
    if config and config.get('clear_screen_between_phases', True):
        os.system('cls')
    
    display_phase_banner("4B", "Retry Timed-Out Conversions (Alternate Method)")
    
    retry_stats = {
        'total': len(timeout_books),
        'converted': 0,
        'merged': 0,
        'failed': 0,
        'failed_conversions': [],
        'failed_merges': []
    }
    
    if not timeout_books:
        print_warn("No timed-out books to retry")
        return retry_stats
    
    # Get source management setting
    source_management = calibre_config.get('source_file_management', 'keep_both') if calibre_config else 'keep_both'
    
    print_step(f"Retrying {len(timeout_books)} timed-out book(s) with alternate method...")
    print()
    print_warn("NOTE: Using calibre-debug with KFX Input plugin")
    print("      This method is more robust for complex/large books")
    print("      Timeout: 5 minutes per book")
    print()
    
    # Process each timed-out book
    for idx, (book_id, title, author) in enumerate(timeout_books, 1):
        print_step(f"Retrying book {idx}/{len(timeout_books)}: '{title}' by {author}")
        
        # Get book info
        if book_id not in book_info:
            print_error(f"  Book info not found for ID {book_id}")
            retry_stats['failed'] += 1
            retry_stats['failed_conversions'].append((book_id, title, author, "Book info not found"))
            print()
            continue
        
        info = book_info[book_id]
        book_path = info['path']
        book_dir = os.path.join(library_path, book_path)
        
        # Find source file
        source_filename, is_kfx_zip = find_source_file_in_directory(book_dir)
        
        if not source_filename:
            error_msg = f"No source file found in {book_path}"
            print_error(f"  {error_msg}")
            retry_stats['failed'] += 1
            retry_stats['failed_conversions'].append((book_id, title, author, error_msg))
            print()
            continue
        
        source_path = os.path.join(book_dir, source_filename)
        epub_filename = os.path.splitext(source_filename)[0] + '.epub'
        epub_path = os.path.join(book_dir, epub_filename)
        
        print(f"  Source: {source_filename}")
        print(f"  Target: {epub_filename}")
        print("  Converting with alternate method...")
        
        # Determine working_dir and raw log path
        script_dir = os.path.dirname(os.path.abspath(__file__))
        user_home = os.path.expanduser("~")
        can_write, _ = check_write_permissions(script_dir)
        working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
        
        # Check if raw logging is enabled
        config = load_config()
        enable_raw_logs = config.get('enable_raw_logs', False) if config else False
        raw_log_path = None
        if enable_raw_logs:
            raw_log_path = get_raw_log_path(working_dir, 'conversion')
        
        # Use alternate conversion method with raw logging
        book_info_str = f"{book_id} - {title} by {author}"
        success, error, error_type = convert_book_alternate_method(
            source_path, 
            epub_path, 
            library_path,
            raw_log_path=raw_log_path,
            book_info=book_info_str
        )
        
        if not success:
            print_error(f"  {error}")
            retry_stats['failed'] += 1
            retry_stats['failed_conversions'].append((book_id, title, author, error))
            print()
            continue
        
        print_ok("  Conversion successful")
        retry_stats['converted'] += 1
        
        # Check if this was a PDF extraction (not EPUB)
        if error_type.get('is_pdf', False):
            # PDF was extracted instead of EPUB
            pdf_path = epub_path.replace('.epub', '.pdf')
            print("  Merging PDF format to Calibre...")
            success, error = add_epub_format_to_calibre(book_id, pdf_path, library_path)
        else:
            # Normal EPUB conversion
            print("  Merging EPUB format to Calibre...")
            success, error = add_epub_format_to_calibre(book_id, epub_path, library_path)
        
        if not success:
            print_error(f"  {error}")
            retry_stats['failed_merges'].append((book_id, title, author, error))
        else:
            print_ok("  EPUB format merged successfully")
            retry_stats['merged'] += 1
            
            # IMPORTANT: Phase 4B uses alternate conversion methods
            # We NEVER delete source files when alternate methods are used
            # This preserves the original files for troubleshooting problematic books
            print("  Keeping source format (alternate method used)")
        
        print()
    
    # Display summary
    print("--------------------------------------------------")
    print_step("Retry Summary:")
    print()
    print_ok(f"Total books retried: {retry_stats['total']}")
    print_ok(f"Successfully converted: {retry_stats['converted']}")
    print_ok(f"Successfully merged to Calibre: {retry_stats['merged']}")
    
    if retry_stats['failed'] > 0:
        print_error(f"Failed conversions: {retry_stats['failed']}")
    
    # Build summary points
    summary_points = [
        f"Retried {retry_stats['total']} timed-out book(s) with alternate method",
        f"Successfully converted: {retry_stats['converted']} book(s)",
        f"Successfully merged to Calibre: {retry_stats['merged']} book(s)"
    ]
    
    if retry_stats['failed'] > 0:
        summary_points.append(f"Failed conversions: {retry_stats['failed']} book(s)")
    
    display_phase_summary("4B", "Retry Timed-Out Conversions", summary_points, pause_seconds=5)
    
    return retry_stats

def attempt_calibre_import(content_dir, script_dir, calibre_already_confirmed=False, extraction_stats=None, working_dir=None):
    """
    Main entry point for Calibre auto-import functionality
    Simplified approach using direct library path
    Returns: tuple (imported_count, book_ids, config) for Phase 4 processing
    
    Args:
        calibre_already_confirmed: If True, skip asking user to close Calibre (already asked in pre-flight)
        extraction_stats: Dict with extraction statistics including failed_books list
        working_dir: Working directory (respects fallback paths)
    """
    # Load config to check clear screen setting
    config = load_config()
    if config and config.get('clear_screen_between_phases', True):
        os.system('cls')
    
    display_phase_banner(3, "Calibre Auto-Import")
    
    try:
        # Load unified config to get calibre_import settings
        unified_config = load_config()
        
        if not unified_config or 'calibre_import' not in unified_config:
            print_warn("Calibre auto-import not configured")
            print()
            return 0
        
        config = unified_config['calibre_import']
        
        # Check if Calibre import is enabled
        if not config.get('enabled', False):
            print_warn("Calibre auto-import is disabled in configuration")
            print()
            return 0
        
        print_step("Configuration validated successfully!")
        
        # Get book count for display
        book_count = get_library_book_count(config['library_path'])
        if book_count is not None:
            print(f"Library path: {config['library_path']} ({book_count} books)")
        else:
            print(f"Library path: {config['library_path']}")
        print()
        
        # Only ask to close Calibre if not already confirmed
        if not calibre_already_confirmed:
            if not warn_close_calibre():
                print_warn("Import cancelled by user")
                print()
                return 0
        
        # Extract list of failed AND skipped ASINs from extraction stats
        exclude_asins = []
        if extraction_stats:
            # Add failed books
            if extraction_stats.get('failed_books'):
                exclude_asins.extend([asin for asin, _, _ in extraction_stats['failed_books']])
            # Add skipped books (previously processed)
            if extraction_stats.get('skipped_books'):
                exclude_asins.extend([asin for asin, _, _ in extraction_stats['skipped_books']])
            
            if exclude_asins:
                failed_count = len(extraction_stats.get('failed_books', []))
                skipped_count = len(extraction_stats.get('skipped_books', []))
                print_warn(f"Excluding {len(exclude_asins)} book(s) from import:")
                if failed_count > 0:
                    print(f"      - {failed_count} failed key extraction")
                if skipped_count > 0:
                    print(f"      - {skipped_count} previously processed (skipped)")
                print()
        
        # Phase 3a: Optional cleanup of KFX-ZIP books
        print_step("[PHASE 3a] Checking for existing KFX-ZIP books...")
        removed_count, cleanup_skipped = cleanup_kfx_zip_books(config['library_path'])
        
        # Silently attempt to reset book ID counter if library is empty
        reset_calibredb_book_id(config['library_path'])
        
        # Import all ebooks (use duplicates flag if cleanup was skipped)
        # Pass exclude_asins to skip books that failed key extraction
        # Returns dict with: total, success, failed, timed_out, book_ids, failed_books, timed_out_books
        results = import_all_azw_files(content_dir, config['library_path'], use_duplicates=cleanup_skipped, exclude_asins=exclude_asins, working_dir=working_dir)
        
        # Extract stats from results dict
        imported_count = results['success']
        book_ids = results['book_ids']
        
        # Display results
        display_import_results(results)
        
        # Build summary points
        summary_points = [
            f"Imported {imported_count} ebook(s) to Calibre library",
            f"Books added with IDs: {', '.join(book_ids) if book_ids else 'None'}",
            "DeDRM plugin automatically processed all imports",
            "Books are now available in Calibre"
        ]
        
        # Add timeout/failure info to summary if applicable
        if results.get('timed_out', 0) > 0:
            summary_points.append(f"Timed out: {results['timed_out']} book(s) - continuing with remaining books")
        
        if results.get('failed', 0) > 0:
            summary_points.append(f"Failed: {results['failed']} book(s) - check error messages above")
        
        display_phase_summary(3, "Calibre Auto-Import", summary_points, pause_seconds=5)
        
        # Return tuple: (imported_count, book_ids, config, results)
        return (imported_count, book_ids, config, results)
        
    except KeyboardInterrupt:
        print()
        print_warn("Calibre auto-import cancelled by user")
        print()
        return 0
    except Exception as e:
        print_error(f"Calibre auto-import failed: {e}")
        print()
        return 0

def main():
    # === Pre-Flight Configuration Total Steps ===
    TOTAL_STEPS = 8
    
    # === OS CHECK: Windows Only ===
    current_os = platform.system()
    
    if current_os != 'Windows':
        os.system('clear' if current_os != 'Windows' else 'cls')
        print()
        print_error("=" * 70)
        print_error("THIS SCRIPT ONLY WORKS ON WINDOWS!")
        print_error("=" * 70)
        print()
        print_error(f"Current OS detected: {current_os}")
        print()
        print("This script is designed for Windows 11 and relies on Windows-specific:")
        print("  - File paths (C:\\Users\\...\\AppData\\...)")
        print("  - Kindle for PC installation locations")
        print("  - Calibre configuration file locations (AppData\\Roaming\\calibre)")
        print()
        print("=" * 70)
        print_step("MANUAL EXTRACTION INSTRUCTIONS")
        print("=" * 70)
        print()
        print("To manually extract Kindle keys, run Satsuoni's KFXKeyExtractor28.exe:")
        print()
        print("  KFXKeyExtractor28.exe <content_dir> <output_key> <output_k4i>")
        print()
        print("Example:")
        print('  KFXKeyExtractor28.exe "C:\\Users\\YourName\\Documents\\My Kindle Content" "kindlekey.txt" "kindlekey.k4i"')
        print()
        print("Where:")
        print("  <content_dir>  = Path to your Kindle content folder")
        print("  <output_key>   = Output file for key (e.g., kindlekey.txt)")
        print("  <output_k4i>   = Output file for account data (e.g., kindlekey.k4i)")
        print()
        print("=" * 70)
        print_step("NEED HELP ON NON-WINDOWS SYSTEMS?")
        print("=" * 70)
        print()
        print("For assistance running this on Mac/Linux or other non-Windows systems,")
        print("please visit Satsuoni's GitHub Repository and request help from the")
        print("author directly:")
        print()
        print_colored("  https://github.com/Satsuoni/DeDRM_tools/discussions", 'cyan')
        print()
        print("=" * 70)
        print()
        input("Press Enter to exit...")
        return 1
    
    # Define paths - use script directory and current user instead of hardcoded paths
    script_dir = os.path.dirname(os.path.abspath(__file__))
    fixed_dir = script_dir
    user_home = os.path.expanduser("~")
    extractor = os.path.join(fixed_dir, "KFXKeyExtractor28.exe")
    
    # Initialize working_dir as None - will be set during validation
    working_dir = None
    
    # Define initial paths (will be updated after validation if using fallback)
    keys_dir = os.path.join(fixed_dir, "Keys")
    output_key = os.path.join(keys_dir, "kindlekey.txt")
    output_k4i = os.path.join(keys_dir, "kindlekey.k4i")
    
    dedrm_json = os.path.join(user_home, "AppData", "Roaming", "calibre", "plugins", "dedrm.json")
    reference_json = os.path.join(fixed_dir, "dedrm  filled.json")
    
    # Define backup paths (will be updated after validation if using fallback)
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    backups_dir = os.path.join(fixed_dir, "backups")
    backup_json = os.path.join(backups_dir, f"dedrm_backup_{timestamp}.json")
    
    os.system('cls')  # Clear screen
    print_banner_and_version()
    print("==================================================")
    print("Phase 1: Key Extraction (Plugin-Compatible)")
    print("Phase 2: DeDRM Plugin Auto-Configuration")
    print("Phase 3: Calibre Auto-Import")
    print("Phase 4: Imported eBooks to EPUB Conversion")
    print("==================================================")
    print()

    try:
        # Cleanup any leftover temporary Kindle installations
        cleanup_temp_kindle()
        
        # Cleanup any leftover temp_extraction folder
        cleanup_temp_extraction()
        
        # Cleanup any leftover staging folder
        cleanup_temp_staging()
        
        # =====================================================================
        # PRE-FLIGHT: VALIDATE ALL REQUIREMENTS
        # =====================================================================
        print_step("Validating system requirements...")
        print()
        
        validation_report = validate_all_requirements(script_dir, user_home)
        
        if not display_validation_results(validation_report):
            return 1  # User cancelled or critical components missing
        
        # Use the validated working directory for all operations
        working_dir = validation_report['working_dir']
        is_using_fallback = validation_report['is_fallback']
        
        # If using fallback, update all path variables
        if is_using_fallback:
            fixed_dir = working_dir
            keys_dir = os.path.join(working_dir, "Keys")
            backups_dir = os.path.join(working_dir, "backups")
            
            # Update file paths
            output_key = os.path.join(keys_dir, "kindlekey.txt")
            output_k4i = os.path.join(keys_dir, "kindlekey.k4i")
            backup_json = os.path.join(backups_dir, f"dedrm_backup_{timestamp}.json")
        
        # CRITICAL: Always ensure directories exist before Phase 1 extraction
        # This must happen regardless of fallback status to prevent write failures
        os.makedirs(keys_dir, exist_ok=True)
        os.makedirs(backups_dir, exist_ok=True)
        
        # =====================================================================
        # PRE-FLIGHT: CHECK FOR SAVED CONFIGURATION
        # =====================================================================
        saved_config = load_config()
        
        if saved_config:
            # Check if configuration version matches current script version
            is_valid, config_version, current_version = check_config_version(saved_config)
            
            if not is_valid:
                # Version mismatch detected - force reconfiguration
                print_error("=" * 70)
                print_error("CONFIGURATION VERSION MISMATCH DETECTED!")
                print_error("=" * 70)
                print()
                print_warn(f"Saved Configuration Version: {config_version}")
                print_warn(f"Current Script Version:      {current_version}")
                print()
                print("The configuration format has changed and requires reconfiguration.")
                print("This ensures compatibility with new features and settings.")
                print()
                input("Press Enter to start the configuration wizard...")
                print()
                saved_config = configure_pre_flight_wizard(user_home, TOTAL_STEPS)
            else:
                # Configuration exists and version matches - show options with timer
                action = prompt_config_action_with_timer(saved_config)
                
                if action == 'quit':
                    print_warn("Script cancelled by user")
                    return 0
                elif action == 'reconfigure':
                    print_step("Starting reconfiguration wizard...")
                    saved_config = configure_pre_flight_wizard(user_home, TOTAL_STEPS)
                # else action == 'use', proceed with saved config
        else:
            # First run - show wizard
            print_step("First run detected - starting configuration wizard...")
            print()
            saved_config = configure_pre_flight_wizard(user_home, TOTAL_STEPS)
        
        # Check if Calibre auto-import is enabled and ask to close Calibre NOW
        calibre_ready = False
        calibre_import_config = saved_config.get('calibre_import', {})
        if isinstance(calibre_import_config, dict) and calibre_import_config.get('enabled', False):
            print_step("Calibre Auto-Import is enabled")
            print("--------------------------------------------------")
            print()
            if not warn_close_calibre():
                print_warn("Calibre import will be skipped")
                calibre_ready = False
            else:
                calibre_ready = True
            print()
        
        # Use configured paths
        content_dir = saved_config.get('kindle_content_path')
        if not content_dir:
            # Fallback to interactive prompt if not in config
            default_content_dir = os.path.join(user_home, "Documents", "My Kindle Content")
            content_dir = get_kindle_content_path(default_content_dir)
        
        # === AUTO-LAUNCH KINDLE (if enabled) ===
        if saved_config.get('auto_launch_kindle', False):
            # Clear screen if configured
            if saved_config.get('clear_screen_between_phases', True):
                os.system('cls')
            
            print()
            print_colored("═" * 70, 'cyan')
            print_colored(f"║{'AUTO-LAUNCH KINDLE':^68}║", 'cyan')
            print_colored(f"║{f'Script Version: {SCRIPT_VERSION}':^68}║", 'cyan')
            print_colored("═" * 70, 'cyan')
            print()
            
            # Launch Kindle and wait for it to close
            success, error_msg = launch_and_wait_for_kindle()
            
            if not success:
                print_error(f"Failed to launch Kindle: {error_msg}")
                print_warn("Continuing with key extraction anyway...")
                print()
            
            # Scan for books after Kindle closes
            print_step("Scanning for downloaded books...")
            book_folders = scan_kindle_content_directory(content_dir)
            
            if not book_folders:
                print_error("No books detected in Kindle content directory!")
                print()
                print_warn("Please ensure you have downloaded books in Kindle before running this script.")
                print(f"Content directory: {content_dir}")
                print()
                print("Script will now exit.")
                input("Press Enter to exit...")
                return 1
            
            print_ok(f"Found {len(book_folders)} book(s) ready for processing")
            print()
            
            # Pause before continuing
            print_step("Continuing to key extraction in 3 seconds...")
            time.sleep(3)
        
        # === PHASE 1: KEY EXTRACTION ===
        # Clear screen if configured
        if saved_config.get('clear_screen_between_phases', True):
            os.system('cls')
        
        display_phase_banner(1, "Key Extraction")

        # Cleanup previous files from both possible locations (failsafe)
        # Previous run might have used different location based on write permissions
        cleanup_locations = [
            os.path.join(script_dir, "Keys"),  # Script directory
            os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder", "Keys")  # Fallback
        ]
        
        for location in cleanup_locations:
            for filename in ["kindlekey.txt", "kindlekey.k4i"]:
                file_path = os.path.join(location, filename)
                if os.path.exists(file_path):
                    os.remove(file_path)
                    print_ok(f"Previous {filename} deleted from {os.path.basename(location)}")

        print()
        
        # Extract keys using the extractor (since we need both txt and k4i files)
        print_step("Extracting Kindle keys...")
        print("--------------------------------------------------")
        print()
        
        result = extract_keys_using_extractor(extractor, content_dir, output_key, output_k4i, working_dir=working_dir)
        
        # Check if user quit during extraction
        if result == 0:
            return 0
        
        success, dsn, tokens, extraction_stats, content_dir = result
        
        print("--------------------------------------------------")
        print()
        
        # Check if extraction was successful
        if not success:
            if extraction_stats['total'] == 0:
                print_error("No books found in content directory!")
                return 1
            elif extraction_stats['success'] == 0:
                print_error("Key extraction failed! No keys were extracted from any books.")
                print_error(f"Failed: {extraction_stats['failed']} book(s)")
                if extraction_stats.get('failed_books'):
                    print()
                    print_warn("Failed books:")
                    for asin, title, error_msg in extraction_stats['failed_books'][:5]:
                        print(f"  - {asin}: {error_msg}")
                    if len(extraction_stats['failed_books']) > 5:
                        print(f"  ... and {len(extraction_stats['failed_books']) - 5} more")
                return 1
        
        # Check if no books were processed (all previously processed scenario)
        if success and extraction_stats['success'] == 0 and extraction_stats.get('skipped', 0) > 0:
            print_ok("All books have been previously processed - script completed successfully")
            print()
            return 0
        
        # Display extraction results
        print_ok("Keys successfully extracted:")
        print(f"   - {output_key}")
        print(f"   - {output_k4i}")
        print()
        
        # Show extraction statistics
        if extraction_stats['total'] > 0:
            print_step("Extraction Statistics:")
            print(f"   Total books found: {extraction_stats['total']}")
            print(f"   Successfully extracted: {extraction_stats['success']}")
            if extraction_stats['failed'] > 0:
                print_warn(f"   Failed: {extraction_stats['failed']}")
        
        print()
        
        # Prevent Kindle auto-updates (if Kindle is installed)
        prevent_kindle_auto_update()
        
        # Display Phase 1 summary
        summary_points = [
            f"Processed {extraction_stats['total']} book(s) for key extraction",
            f"Successfully extracted keys from {extraction_stats['success']} book(s)",
            f"Generated kindlekey.txt at {output_key}",
            f"Generated kindlekey.k4i at {output_k4i}",
            "Kindle auto-update prevention configured"
        ]
        
        if extraction_stats.get('skipped', 0) > 0:
            summary_points.append(f"Skipped: {extraction_stats['skipped']} book(s) previously processed")
        
        if extraction_stats['failed'] > 0:
            summary_points.append(f"Failed extractions: {extraction_stats['failed']} book(s) - see log for details")
        
        display_phase_summary(1, "Key Extraction", summary_points, pause_seconds=5)

        # === PHASE 2: DEDRM PLUGIN CONFIGURATION ===
        # Clear screen if configured
        if saved_config.get('clear_screen_between_phases', True):
            os.system('cls')
        
        display_phase_banner(2, "DeDRM Plugin Configuration")

        # Create backup if dedrm.json exists
        if os.path.exists(dedrm_json):
            print_step("Creating backup of existing dedrm.json...")
            shutil.copy2(dedrm_json, backup_json)
            print_ok(f"Backup created: {backup_json}")
        else:
            print_warn("No existing dedrm.json found - will create new one.")

        # Process the k4i file to create kindle key
        print_step("Processing kindlekey.k4i...")
        kindle_key = create_kindle_key_from_k4i(output_k4i, dsn, tokens)
        
        if not kindle_key:
            print_error("Failed to process k4i file!")
            return 1
            
        print_ok("Kindle key data processed successfully.")

        # Create the dedrm configuration
        print_step("Creating DeDRM configuration...")
        dedrm_config = create_dedrm_config(kindle_key, output_key, reference_json)

        # Write the JSON using the same method as the plugin: json.dump() with indent=2
        print_step("Writing dedrm.json with exact plugin formatting...")
        
        # Ensure the directory exists
        os.makedirs(os.path.dirname(dedrm_json), exist_ok=True)
        
        with open(dedrm_json, 'w') as f:
            json.dump(dedrm_config, f, indent=2)

        print_ok("DeDRM configuration updated successfully!")
        print_ok("Updated key: kindlekey")
        print_ok(f"Set extra key file: {output_key}")
        print()

        # Final verification
        print_step("Verifying configuration...")
        print()

        with open(dedrm_json, 'r') as f:
            dedrm_verify = json.load(f)
        
        # Count kindle keys
        key_count = len(dedrm_verify.get("kindlekeys", {}))
        
        # Get hide_sensitive flag
        hide_sensitive = saved_config.get('hide_sensitive_info', False)
        
        # Read voucher keys from kindlekey.txt
        voucher_keys = []
        if os.path.exists(output_key):
            try:
                with open(output_key, 'r') as f:
                    voucher_keys = [line.strip() for line in f if line.strip()]
            except Exception:
                pass
        
        # Table header (extra wide to fit 88-char base64 New Secret)
        print("┌─────────────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────┐")
        print("│ Configuration Item          │ Value                                                                                          │")
        print("├─────────────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────┤")
        
        # Total keys
        print(f"│ Total Kindle Keys           │ {str(key_count):<94} │")
        
        # Extra key file path
        extra_key_path = dedrm_verify.get('kindleextrakeyfile', 'Not set')
        if len(extra_key_path) > 94:
            extra_key_path = extra_key_path[:91] + "..."
        print(f"│ Extra Key File Path         │ {extra_key_path:<94} │")
        
        # Show key details with obfuscation if enabled
        if "kindlekeys" in dedrm_verify and "kindlekey" in dedrm_verify["kindlekeys"]:
            key_data = dedrm_verify["kindlekeys"]["kindlekey"]
            
            # DSN (full width, no truncation unless obfuscated)
            dsn_value = key_data.get('DSN', 'Not found')
            if hide_sensitive and dsn_value != 'Not found':
                dsn_value = obfuscate_sensitive(dsn_value)
            print(f"│ DSN                         │ {dsn_value:<94} │")
            
            # DSN_clear (if available)
            dsn_clear = key_data.get('DSN_clear', '')
            if dsn_clear:
                if hide_sensitive:
                    dsn_clear = obfuscate_sensitive(dsn_clear)
                print(f"│ DSN Clear                   │ {dsn_clear:<94} │")
            
            # Tokens (full width, no truncation unless obfuscated)
            tokens_value = key_data.get('kindle.account.tokens', '')
            if tokens_value:
                if hide_sensitive:
                    tokens_value = obfuscate_sensitive(tokens_value)
                print(f"│ Tokens                      │ {tokens_value:<94} │")
            
            # Show sample of secrets if available (obfuscated if needed)
            clear_old_secrets = key_data.get('kindle.account.clear_old_secrets', [])
            new_secrets = key_data.get('kindle.account.new_secrets', [])
            
            if clear_old_secrets:
                sample_clear_old = clear_old_secrets[0] if isinstance(clear_old_secrets, list) else str(clear_old_secrets)
                if hide_sensitive:
                    sample_clear_old = obfuscate_sensitive(sample_clear_old)
                print(f"│ Old Secret                  │ {sample_clear_old:<94} │")
            
            if new_secrets:
                sample_new_secret = new_secrets[0] if isinstance(new_secrets, list) else str(new_secrets)
                if hide_sensitive:
                    sample_new_secret = obfuscate_sensitive(sample_new_secret)
                print(f"│ New Secret                  │ {sample_new_secret:<94} │")
        
        # Voucher keys count at bottom
        print(f"│ Voucher Keys Count          │ {str(len(voucher_keys)):<94} │")
        
        # Table footer
        print("└─────────────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────┘")
        print()
        
        print_ok("Configuration verified successfully!")
        print()
        
        # Display Phase 2 summary
        summary_points = [
            "Processed kindlekey.k4i and created Kindle key data",
            "Updated DeDRM plugin configuration (dedrm.json)",
            f"Set extra key file path: {output_key}",
            "Added account keys to plugin database",
            f"Created configuration backup: {backup_json}",
            "Configuration verified successfully"
        ]
        
        display_phase_summary(2, "DeDRM Plugin Configuration", summary_points, pause_seconds=5)
        
        # === PHASE 3: CALIBRE AUTO-IMPORT ===
        imported_count = 0
        converted_count = 0
        import_results = None
        conversion_stats = None
        result = attempt_calibre_import(content_dir, script_dir, calibre_already_confirmed=calibre_ready, extraction_stats=extraction_stats, working_dir=working_dir)
        
        # Handle result - could be 0 (skipped), or tuple (imported_count, book_ids, config, results)
        if result == 0:
            imported_count = 0
        elif isinstance(result, tuple):
            if len(result) == 4:
                # New format with results dict
                imported_count, book_ids, config, import_results = result
            else:
                # Old format without results dict (backward compatibility)
                imported_count, book_ids, config = result
                import_results = None
            
            # === PHASE 4: KFX TO EPUB CONVERSION ===
            if config.get('convert_to_epub', False) and book_ids:
                conversion_stats = process_book_conversions(config['library_path'], book_ids, config, working_dir=working_dir)
                converted_count = conversion_stats['merged']
                
                # === PHASE 4B: RETRY FAILED CONVERSIONS (INCLUDING TIMEOUTS) ===
                # Collect ALL failed books (timeouts + other failures)
                all_failed_books = []
                
                # Add timeout books
                if conversion_stats.get('timeout_books'):
                    all_failed_books.extend(conversion_stats['timeout_books'])
                
                # Add other failed conversions (non-timeout failures)
                if conversion_stats.get('failed_conversions'):
                    for book_id, title, author, error_msg in conversion_stats['failed_conversions']:
                        # Only add if not already in timeout_books
                        if not any(b[0] == book_id for b in conversion_stats.get('timeout_books', [])):
                            all_failed_books.append((book_id, title, author))
                
                if len(all_failed_books) > 0:
                    # Get book info for retry
                    book_info = query_book_info_from_db(config['library_path'], book_ids)
                    
                    # Clear screen if configured
                    if saved_config.get('clear_screen_between_phases', True):
                        os.system('cls')
                    
                    print()
                    print_colored("═" * 70, 'cyan')
                    print_colored(f"║{'FAILED CONVERSION RETRY PROMPT':^68}║", 'cyan')
                    print_colored(f"║{f'Script Version: {SCRIPT_VERSION}':^68}║", 'cyan')
                    print_colored("═" * 70, 'cyan')
                    print()
                    
                    print_warn(f"{len(all_failed_books)} book(s) failed during conversion")
                    print()
                    print("These books failed to convert using the standard method.")
                    print("They can be retried using an alternate, more robust conversion method.")
                    print()
                    print("Failed books:")
                    for book_id, title, author in all_failed_books[:5]:
                        print(f"  - {title} by {author}")
                    if len(all_failed_books) > 5:
                        print(f"  ... and {len(all_failed_books) - 5} more")
                    print()
                    print("Alternate method details:")
                    print("  - Uses calibre-debug with KFX Input plugin")
                    print("  - More robust for complex/large books")
                    print("  - 5-minute timeout per book (vs 3-minute standard)")
                    print("  - Streams debug output for better visibility")
                    print()
                    
                    while True:
                        choice = input("Retry failed books with alternate method? (Y/N) [Y]: ").strip().upper()
                        if choice == '':
                            choice = 'Y'  # Default to Yes
                        if choice in ['Y', 'N']:
                            break
                        print_error("Please enter Y or N")
                    
                    if choice == 'Y':
                        print()
                        retry_stats = process_timeout_retries(
                            config['library_path'],
                            all_failed_books,
                            book_info,
                            config
                        )
                        
                        # Update converted count with retry successes
                        converted_count += retry_stats['merged']
                        
                        # Store retry stats for final summary
                        conversion_stats['retry_stats'] = retry_stats
                    else:
                        print()
                        print_warn("Skipping retry - failed books will remain unconverted")
                        print()
        else:
            imported_count = result if result is not None else 0
        
        # Clear screen only if configured to do so
        if saved_config.get('clear_screen_between_phases', True):
            os.system('cls')
        
        print("==================================================")
        print_done("SUCCESS! Complete automation finished!")
        print("==================================================")
        print()
        print_ok("What was accomplished:")
        print("  + Extracted Kindle keys using plugin-compatible method")
        print("  + Generated kindlekey.txt (voucher keys)")
        print("  + Generated kindlekey.k4i (account data)")
        print("  + Updated DeDRM plugin configuration automatically")
        print("  + Used exact same JSON generation as the plugin")
        print("  + Set extra key file path in plugin")
        print("  + Added account keys to plugin database")
        print("  + Created configuration backup for safety")
        if imported_count > 0:
            print(f"  + Imported {imported_count} ebook(s) to Calibre")
        if converted_count > 0:
            print(f"  + Converted and merged {converted_count} ebook(s) to EPUB format")

        # Cleanup staging directory after all phases complete (if it was created)
        staging_dir = os.path.join(working_dir, "temp_kindle_content")
        if os.path.exists(staging_dir):
            # Use robust cleanup that handles OneDrive locks
            print()
            print_step("Cleaning Staging Directory/Files")
            rmtree_with_retry(staging_dir)
        
        # Show extraction issues if any
        if extraction_stats and (extraction_stats.get('skipped', 0) > 0 or extraction_stats.get('failed', 0) > 0):
            print()
            if extraction_stats.get('skipped', 0) > 0:
                print_warn(f"Extraction Skipped: {extraction_stats['skipped']} book(s) previously processed")
                if extraction_stats.get('skipped_books'):
                    print("      Skipped books:")
                    for asin, title, _ in extraction_stats['skipped_books'][:5]:
                        print(f"        - {asin} - {title}")
                    if len(extraction_stats['skipped_books']) > 5:
                        print(f"        ... and {len(extraction_stats['skipped_books']) - 5} more")
            
            if extraction_stats.get('failed', 0) > 0:
                if extraction_stats.get('skipped', 0) > 0:
                    print()  # Add spacing between skipped and failed sections
                print_warn(f"Extraction Issues: {extraction_stats['failed']} book(s) failed key extraction")
                if extraction_stats.get('failed_books'):
                    print("      Failed books:")
                    for asin, title, error_msg in extraction_stats['failed_books']:
                        print(f"        - {asin} - {title}")
        
        # Show import/conversion issues if any
        if import_results and isinstance(import_results, dict):
            # Show skipped books (duplicates) with book titles
            if import_results.get('skipped', 0) > 0:
                print()
                # Print header in RED and BOLD
                print(f"\033[1m\033[91m[!] Import Issues: {import_results['skipped']} book(s) already exist in database\033[0m")
                if import_results.get('skipped_books'):
                    print("      Skipped books:")
                    for asin, title in import_results['skipped_books']:
                        print(f"        - {asin} - {title}")
            
            if import_results.get('failed', 0) > 0:
                print()
                print_warn(f"Import Issues: {import_results['failed']} book(s) failed to import")
                if import_results.get('failed_books'):
                    print("      Failed books:")
                    for book_name, _ in import_results['failed_books'][:5]:
                        print(f"        - {book_name}")
                    if len(import_results['failed_books']) > 5:
                        print(f"        ... and {len(import_results['failed_books']) - 5} more")
            
            if import_results.get('timed_out', 0) > 0:
                print()
                print_warn(f"Import Timeouts: {import_results['timed_out']} book(s) timed out")
                if import_results.get('timed_out_books'):
                    print("      Timed out books:")
                    for book_name in import_results['timed_out_books'][:5]:
                        print(f"        - {book_name}")
                    if len(import_results['timed_out_books']) > 5:
                        print(f"        ... and {len(import_results['timed_out_books']) - 5} more")
        
        if conversion_stats and isinstance(conversion_stats, dict):
            if conversion_stats.get('failed', 0) > 0:
                print()
                print_warn(f"Conversion Issues: {conversion_stats['failed']} book(s) failed to convert")
                if conversion_stats.get('failed_conversions'):
                    print("      Failed conversions:")
                    for book_id, title, author, _ in conversion_stats['failed_conversions'][:5]:
                        print(f"        - {title} by {author}")
                    if len(conversion_stats['failed_conversions']) > 5:
                        print(f"        ... and {len(conversion_stats['failed_conversions']) - 5} more")
            
            if conversion_stats.get('skipped_kfx_zip', 0) > 0:
                print()
                print_warn(f"Conversion Skipped: {conversion_stats['skipped_kfx_zip']} .kfx-zip file(s) skipped")
            
            # Show retry results if Phase 4B was executed
            if conversion_stats.get('retry_stats'):
                retry_stats = conversion_stats['retry_stats']
                print()
                print_ok(f"Retry Results (Phase 4B):")
                print(f"      Retried: {retry_stats['total']} timed-out book(s)")
                print(f"      Successfully converted: {retry_stats['converted']} book(s)")
                print(f"      Successfully merged: {retry_stats['merged']} book(s)")
                if retry_stats.get('failed', 0) > 0:
                    print_warn(f"      Still failed: {retry_stats['failed']} book(s)")
        
        # Determine if we should pause based on errors and skip_phase_pauses setting
        has_errors = False
        if extraction_stats and extraction_stats.get('failed', 0) > 0:
            has_errors = True
        if import_results and isinstance(import_results, dict):
            if import_results.get('skipped', 0) > 0 or import_results.get('failed', 0) > 0 or import_results.get('timed_out', 0) > 0:
                has_errors = True
        if conversion_stats and isinstance(conversion_stats, dict):
            if conversion_stats.get('failed', 0) > 0 or conversion_stats.get('skipped_kfx_zip', 0) > 0:
                has_errors = True
        
        # Always pause if there are errors, otherwise respect skip_phase_pauses setting
        skip_pauses = saved_config.get('skip_phase_pauses', False) if saved_config else False
        should_pause = has_errors or not skip_pauses
        
        if should_pause:
            print()
            print("Press Any key to continue...")
            msvcrt.getch()
        
        os.system('cls')
        print()
        print_banner_and_version()
        print("For the latest version of this script and updates, visit:")
        print_colored("https://techy-notes.com/blog/dedrm-v10-0-14-tutorial", 'cyan')
        print()
        print("Watch the video tutorial on YouTube:")
        print_colored("https://www.youtube.com/watch?v=pkii6EQEeGs", 'cyan')
        print()
        print_warn("Please subscribe to the YouTube channel!")
        print("Your support and appreciation is greatly valued.")
        print()
        print("If you'd like to show extra support this Script, consider buying me a Beer!:")
        print_colored("https://buymeacoffee.com/jadehawk", 'cyan')
        print()
        print("--------------------------------------------------")
        print_step("CREDITS")
        print("--------------------------------------------------")
        print()
        print("This script is powered by KFXKeyExtractor28.exe")
        print("Created/Modded by: Satsuoni")
        print()
        print("KFXKeyExtractor is the CORE tool that makes this automation possible.")
        print("It extracts Kindle DRM keys from your Kindle for PC installation,")
        print("enabling the DeDRM process for your purchased ebooks.")
        print()
        print("Visit Satsuoni's GitHub profile:")
        print_colored("https://github.com/Satsuoni", 'cyan')
        print()
        print("For the DeDRM tools repository:")
        print_colored("https://github.com/Satsuoni/DeDRM_tools", 'cyan')
        print()
        print("Thank you, Satsuoni, for creating and maintaining this essential tool!")

    except Exception as e:
        print_error(f"Script failed: {str(e)}")
        import traceback
        print_error(f"Traceback: {traceback.format_exc()}")
        
        # Determine working_dir for cleanup (in case error occurred before validation)
        if working_dir is None:
            # working_dir not set yet - determine it now
            can_write, _ = check_write_permissions(script_dir)
            cleanup_working_dir = script_dir if can_write else os.path.join(user_home, "AppData", "Local", "Kindle_Key_Finder")
        else:
            cleanup_working_dir = working_dir
        
        # Cleanup staging directory on error (if it was created)
        staging_dir = os.path.join(cleanup_working_dir, "temp_kindle_content")
        if os.path.exists(staging_dir):
            # Use robust cleanup that handles OneDrive locks
            rmtree_with_retry(staging_dir)
        
        # Restore backup if it exists
        if os.path.exists(backup_json) and os.path.exists(dedrm_json):
            print_warn("Restoring backup...")
            shutil.copy2(backup_json, dedrm_json)
            print_ok("Backup restored.")
        
        return 1

    return 0

if __name__ == "__main__":
    sys.exit(main())
